"""I store the information of a single PDW block."""

import struct
import math
import numpy as np
from bitstring import BitArray
from ammosreader import logger

class PDW():
    """
    I store information from a single ppdw data block.

    .. automethod:: __init__
    """

    @classmethod
    def polarities(cls):
         return {0: 'Horizontal/Unknown', 1: 'Vertical', 2: 'Counter clockwise', 3: 'Clockwise'}

    @classmethod
    def modulations(cls):
        return {0: 'Unknown', 1: 'Unmodulated', 2: 'FM', 3: 'LFM', 4: 'PSK-2', 5: 'PSK-3', 6: 'PSK-4',
                7: 'PSK-m', 8: 'NLFM', 9: 'SFM', 10: 'TFM', 11: 'Pulse too short'}
    
    @classmethod
    def from_bytes(cls, byte_string):
        """
        I create an instance of class PDW from data body (8 * 32 bits).

        :param byte_string: a byte string containing a single data body read from a ppdw file
        :type byte_string: byte string

        :return: an instance of class PDW with attributes set according to the data of a data body
        :rtype: PDW
        """

        logger.info("from bytes")
        if (len(byte_string) != 32):
            logger.error("Byte count invalid")
            raise TypeError("Byte count invalid")

        parts = struct.unpack('<Q4s4s4s4s4s4s', byte_string)

        nanoseconds = (parts[0])
        unix_time = np.datetime64('now', 'ns').astype(int)
        if nanoseconds >= unix_time:
            raise OverflowError("Timestamp invalid")
        time_of_arrival = nanoseconds

        third_word = bin(int.from_bytes(parts[1], byteorder='little'))
        padding = 32-len(str(third_word)[2:])
        third_word_bit_string = "0" * padding + str(third_word)[2:]
        pdw_format_identifier = int(third_word_bit_string[0:6], 2)
        center_frequency = int(third_word_bit_string[5:32], 2) # Value of zero means unknown
        
        fourth_word = bin(int.from_bytes(parts[2], byteorder='little'))
        padding = 32-len(str(fourth_word)[2:])
        fourth_word_bit_string = "0" * padding + str(fourth_word)[2:]
        is_valid = bool(int(fourth_word_bit_string[0]))
        is_pulse = bool(int(fourth_word_bit_string[1]))
        level_unit = int(fourth_word_bit_string[2])
        signal_start_missing = bool(int(fourth_word_bit_string[3]))
        signal_end_missing = bool(int(fourth_word_bit_string[4]))
        pulse_width = int(fourth_word_bit_string[7:33], 2)

        fifth_word = bin(int.from_bytes(parts[3], byteorder='little'))
        padding = 32-len(str(fifth_word)[2:])
        fifth_word_bit_string = "0" * padding + str(fifth_word)[2:]
        frequency_shift_or_bandwidth = int(fifth_word_bit_string[0:20], 2)
        pulse_level_or_pulse_field_strength = BitArray(bin=fifth_word_bit_string[20:32]).int

        sixth_word = bin(int.from_bytes(parts[4], byteorder='little'))
        padding = 32-len(str(sixth_word)[2:])
        sixth_word_bit_string = "0" * padding + str(sixth_word)[2:]
        region_of_interest = bool(int(sixth_word_bit_string[0]))
        azimuth_confidence = int(sixth_word_bit_string[1:7], 2)
        modulation = int(sixth_word_bit_string[7:12], 2)
        sector = int(sixth_word_bit_string[28:32], 2)

        seventh_word = bin(int.from_bytes(parts[5], byteorder='little'))
        padding = 32-len(str(seventh_word)[2:])
        seventh_word_bit_string = "0" * padding + str(seventh_word)[2:]
        polarity = int(seventh_word_bit_string[0:2], 2)
        df_quality = int(seventh_word_bit_string[2:9], 2)
        elevation = BitArray(bin=seventh_word_bit_string[9:20]).int
        azimuth = int(seventh_word_bit_string[20:32], 2)

        eighth_word = bin(int.from_bytes(parts[5], byteorder='little'))
        padding = 32-len(str(eighth_word)[2:])
        eighth_word_bit_string = "0" * padding + str(eighth_word)[2:]
        channel = int(eighth_word_bit_string[0:4], 2)

        return PDW(time_of_arrival, pdw_format_identifier, center_frequency, is_valid, is_pulse, level_unit,
                   signal_start_missing, signal_end_missing, pulse_width, frequency_shift_or_bandwidth,
                   pulse_level_or_pulse_field_strength, region_of_interest, azimuth_confidence, modulation,
                   sector, polarity, df_quality, elevation, azimuth, channel)

    def __init__(self, time_of_arrival, pdw_format_identifier, center_frequency, is_valid, is_pulse,
                 level_unit, signal_start_missing, signal_end_missing, pulse_width, frequency_shift_or_bandwidth,
                 pulse_level_or_pulse_field_strength, region_of_interest, azimuth_confidence, modulation,
                 sector, polarity, df_quality, elevation, azimuth, channel):
        r"""
        I return an instance of an Pulse Data word.

        :param time_of_arrival: nanoseconds since 1970-01-01 00:00:00
        :type time_of_arrival: Integer
        :param pdw_format: format code
        :type pdw_format: Integer
        :param center_frequency: center frequency in KHz
        :type center_frequency: Integer
        :param is_valid: flag to mark if pdw data body is valid
        :type is_valid: Boolean
        :param is_pulse: flag to mark if pdw data body contains a pulse or a continuous wave signal
        :type is_pulse: Boolean
        :param level_unit: 1 means dBµV - 0 means dBµV/m
        :type level_unit: Integer
        :param signal_start_missing: signal started before time of arrival
        :type signal_start_missing: Boolean
        :param signal_end_missing: signal stops after time of arrival
        :type signal_end_missing: Boolean
        :param pulse_width: pulse width in nanoseconds - Zero if no valid pulse detected
        :type pulse_width: Integer
        :param frequency_shift_or_bandwidth: Value in KHz - Value set to 1048575 means Unknown
        :type frequency_shift_or_bandwidth: Integer
        :param pulse_level_or_pulse_field_strength: Pulse level or Pulse Field Strength depending on level_unit \
         (-200.0...200.0) in tenth degrees / minus 2048 means no valid level detected
        :type pulse_level_or_pulse_field_strength: Integer
        :param region_of_interest: Marks if signal is from region of interest
        :type region_of_interest: Boolean
        :param azimuth_confidence: degree in tenth steps of (0-62) / 63 means confidence unknown
        :type azimuth_confidence: Integer
        :param modulation: type of modulation (e.g. PSK-2, PSK-4, FM etc.)
        :type modulation: String
        :param sector: reference antenna sector (0-15)
        :type sector: Integer
        :param polarity: Horizontal, Vertical, Clockwise, Counter clockwise
        :type polarity: String
        :param df_quality: Direction finding quality in percent (0-100) - Zero means unknown
        :type df_quality: Integer
        :param elevation: elevation of incoming signal (from -90 to 90 degree) in steps tenths degree \
        minus 1024 means unknown
        :type elevation: Integer
        :param azimuth: azimuth of incoming signal (from 0 to 3599 tenth degree) plus 4095 means unknown
        :type azimuth: Integer
        :param channel: detecting channel (0-16) - Zero means unknown
        :type channel: Integer
        :return: An instance of class PDW with attributes set according to the data of a data body
        :rtype: PDW
        """
        self.__time_of_arrival = time_of_arrival #
        self.__pdw_format_identifier = pdw_format_identifier
        self.__center_frequency = center_frequency #
        self.__is_valid = is_valid #
        self.__is_pulse = is_pulse #
        self.__level_unit = level_unit #
        self.__signal_start_missing = signal_start_missing
        self.__signal_end_missing = signal_end_missing
        self.__pulse_width = pulse_width #
        self.__frequency_shift_or_bandwidth = frequency_shift_or_bandwidth #
        self.__pulse_level_or_pulse_field_strength = pulse_level_or_pulse_field_strength #
        self.__region_of_interest = region_of_interest
        self.__azimuth_confidence = azimuth_confidence
        self.__modulation = modulation #
        self.__sector = sector
        self.__polarity = polarity
        self.__df_quality = df_quality #
        self.__elevation = elevation #
        self.__azimuth = azimuth
        self.__channel = channel #

    def __str__(self):
        """
        I return the string representation of myself.

        :rtype: str
        """
        output = ("Time of arrival: " + str(self.time_of_arrival) + "\n" +
                  "PDW Format identifier: " + str(self.pdw_format_identifier) + "\n" +
                  "Center frequency: " + str(self.center_frequency) + " KHz\n")

        if self.is_valid:
            output += "Signal: Valid\n"
        else:
            output += "Signal: Invalid\n"

        if self.is_pulse:
            output += "Signal type: Pulse\n"
        else:
            output += "Signal type: Continuous wave\n"

        # FIXME: use fstrings or another better performance string concat
        if self.level_unit == "dbµV":
            output += "Pulse level: " + str(self.pulse_level_or_pulse_field_strength) + " dbµV\n"
        else:
            output += "Pulse field strength: " + str(self.pulse_level_or_pulse_field_strength) + " dbµV/m\n"

        output += ("Pulse width: " + str(self.pulse_width) + " nanoseconds\n" +
                   "Frequency shift or bandwidth: " + str(self.frequency_shift_or_bandwidth) + " KHz\n")

        if self.region_of_interest:
            output += "Region of interest: Yes\n"
        else:
            output += "Region of interest: No\n"

        if self.azimuth_confidence == None:
            output += "Azimuth confidence: Invalid\n"
        else:
            output += "Azimuth confidence: " + str(self.azimuth_confidence) + " degree\n"

        output += "Modulation: " + str(self.modulation) + "\n"

        if self.sector == None:
            output += "Sector: Unknown\n"
        else:
            output += "Sector:" + str(self.sector) + "\n"

        output += "Polarity: " + str(self.polarity) + "\n"

        output += "DF quality: " + str(self.df_quality) + " %\n"

        if self.elevation == None:
            output += "Elevation: Unknown\n"
        else:
            output += "Elevation: " + str(self.elevation) + " degree\n"

        if self.azimuth == None:
            output += "Azimuth: Unknown\n"
        else:
            output += "Azimuth: " + str(self.azimuth) + " degree\n"

        output += "Channel: " + str(self.channel) + "\n"

        return output

    @property
    def time_of_arrival(self):
        return np.datetime64(self.__time_of_arrival, 'ns')
    
    @property
    def center_frequency(self):
        if self.__center_frequency == 0:
            return None
        return self.__center_frequency

    @property
    def is_valid(self):
        return self.__is_valid

    @property
    def is_invalid(self):
        return not self.__is_valid

    @property
    def is_pulse(self):
        return self.__is_pulse

    @property
    def is_cw(self):
        return not self.__is_pulse

    @property
    def level_unit(self):
        if self.__level_unit == 1:
            return "dbµV"
        else:
            return "dbµV/m"

    @property
    def signal_start_missing(self):
        return self.__signal_start_missing

    @property
    def signal_end_missing(self):
        return self.__signal_end_missing

    @property
    def pulse_width(self):
        if self.__pulse_width == 0:
            return None
        return self.__pulse_width

    @property
    def frequency_shift_or_bandwidth(self):
        if self.__frequency_shift_or_bandwidth == 1048575:
            return None
        return self.__frequency_shift_or_bandwidth

    @property
    def pulse_level_or_pulse_field_strength(self):
        if self.__pulse_level_or_pulse_field_strength == -2048:
            return None
        return (self.__pulse_level_or_field_strength / 10) 

    @property
    def azimuth_confidence(self):
        if azimuth_confidence == 63:
            return None
        return (azimuth_confidence / 10)

    @property
    def modulation(self):
        return PDW.modulations().get(self.__modulation, PDW.modulations()[0])
    
    @property
    def region_of_interest(self):
        return self.__region_of_interest

    @property
    def sector(self):
        if self.__sector == 0:
            return None
        return self.__sector

    @property
    def polarity(self):
        return PDW.polarities().get(self.__polarity, PDW.polarities()[0])

    @property
    def elevation(self):
        if self.__elevation == -1024:
            return None
        return self.__elevation / 10

    @property
    def azimuth(self):
        if self.__azimuth == 4095:
            return None
        return self.__azimuth / 10

    @property
    def channel(self):
        return self.__channel

    @property
    def pdw_format_identifier(self):
        return self.__pdw_format_identifier
    
    def to_datacrunch_json(self):
        import uuid
        return {'TOA': int(self.time_of_arrival),
                'FORMAT': self.pdw_format_identifier,
                'CENTER_FREQUENCY': self.center_frequency,
                'VALID_FLAG': self.is_valid,
                'PULSE_FLAG': self.is_pulse,
                'LU_FLAG': self.level_unit,
                'SNS_FLAG': self.signal_start_missing,
                'SNE_FLAG': self.signal_end_missing,
                'LEVEL': self.pulse_level_or_pulse_field_strength,
                'PULSE_WIDTH': self.pulse_width,
                'BANDWIDTH': self.frequency_shift_or_bandwidth,
                'IN_OUT_FLAG': self.region_of_interest,
                'DF_CONFIDENCE': self.azimuth_confidence,
                'MODULATION': self.modulation,
                'SECTOR': self.sector,
                'POL': self.polarity,
                'DF_QUALITY': self.df_quality,
                'ELEVATION': self.elevation,
                'AZIMUTH': self.azimuth,
                'CHANNEL': self.channel,
                'UUID': uuid.uuid4()
                }        

    def to_json(self):
        return {'TIME OF ARRIVAL': self.time_of_arrival,
                'FORMATIDENTIFIER': self.pdw_format_identifier,
                'CENTER FREQUENCY': self.center_frequency,
                'VALID': self.is_valid,
                'PULSE': self.is_pulse,
                'PULSELEVEL': self.pulse_level_or_pulse_field_strength,
                'PULSEWIDTH': self.pulse_width,
                'BANDWIDTH': self.frequency_shift_or_bandwidth,
                'REGIONOFINTEREST': self.region_of_interest,
                'AZIMUTHCONFIDENCE': self.azimuth_confidence,
                'MODULATION': self.modulation,
                'SECTOR': self.sector,
                'POLARITY': self.polarity,
                'DFQUALITY': self.df_quality,
                'ELEVATION': self.elevation,
                'AZIMUTH': self.azimuth,
                'CHANNEL': self.channel
                }

if __name__ == '__main__':
    pass
