"""I provide a specialized Ammos Reader for audio data."""

from ammosreader.AbstractAmmosReader import AbstractAmmosReader
from ammosreader.AmmosGlobalFrameBody import AmmosGlobalFrameBody
from ammosreader.AmmosAudioDataHeader import AmmosAudioDataHeader
from ammosreader.AmmosExtendedAudioDataHeader import AmmosExtendedAudioDataHeader
from ammosreader.AmmosAudioDataBlock import AmmosAudioDataBlock


class AmmosAudioReader(AbstractAmmosReader):
    """I read the audio data embedded in an R&S AMMOS recording."""

    STANDARD_AUDIO_DATA_HEADER_SIZE = 36  # 9 words
    EXTENDED_AUDIO_DATA_HEADER_SIZE = 44  # 11 words

    def __init__(self, file_name):
        """
        I return an instance of AmmosAudioReader initialized with a given file name.

        :param file_name: the file to read from
        :type file_name: str
        """
        super().__init__(file_name)

    def read_next_global_frame_body_data_header(self):
        """
        I return the next global frame body data header from current position in file.

        :param data_header_size: the number of bytes to read
        :type data_header_size: int
        :return: the next Ammos Audio Data header or None if incomplete
        :rtype: AmmosAudioDataHeader
        """
        bytes = self.ammos_file.read(AmmosAudioReader.STANDARD_AUDIO_DATA_HEADER_SIZE)

        # print("\nReading global frame body standard data header\n")
        if ((not bytes) or (len(bytes) < AmmosAudioReader.STANDARD_AUDIO_DATA_HEADER_SIZE)):
            print("Can not read all", AmmosAudioReader.STANDARD_AUDIO_DATA_HEADER_SIZE,
                  "bytes of global frame body data header")
            return None
        return AmmosAudioDataHeader.from_bytes(bytes)

    def read_next_global_frame_body_extended_data_header(self):
        """
        I return the next global frame body extended data header from current position in file.

        :return: the next Ammos Audio Extended Data header or None if incomplete
        :rtype: AmmosExtendedAudioDataHeader
        """
        bytes = self.ammos_file.read(AmmosAudioReader.EXTENDED_AUDIO_DATA_HEADER_SIZE)

        if ((not bytes) or (len(bytes) < AmmosAudioReader.EXTENDED_AUDIO_DATA_HEADER_SIZE)):
            print("Can not read all ", AmmosAudioReader.EXTENDED_AUDIO_DATA_HEADER_SIZE,
                  " bytes of global frame extended data header")
            return None
        return AmmosExtendedAudioDataHeader.from_bytes(bytes)

    def read_next_audio_data_body(self, sample_count, channel_count, sample_size):
        """
        I return the next audio data read from current position in file.

        :param sample_count: the number of samples per channel inside data body
        :type sample_count: int

        :param channel_count: number of channels (e.g. mono, stereo or even more)
        :type channel_count: int

        :param sample_size: sample size in bytes (1, 2 or 4 bytes)
        :type sample_size: int

        :return: the next audio data or None if incomplete
        :rtype: bytes
        """
        # FIXME: Describe the parameters better

        total = sample_count*channel_count*sample_size

        byte_string = self.ammos_file.read(total)

        if len(byte_string) != total:
            print("Can not read all", total, "bytes of data body")
            return None
        # print([hex(c) for c in byte_string])
        return AmmosAudioDataBlock(byte_string, channel_count, sample_count, sample_size)

    def read_next_global_frame_body(self, data_header_length):
        """
        I return the next global frame body read from current position in file.

        :param global_frame_header:
        """
        audio_data_header = None

        if data_header_length == AmmosAudioReader.STANDARD_AUDIO_DATA_HEADER_SIZE:
            print("Read standard data header")
            audio_data_header = self.read_next_global_frame_body_data_header()

        print("Data header length", data_header_length)
        if data_header_length == AmmosAudioReader.EXTENDED_AUDIO_DATA_HEADER_SIZE:
            print("Read extended data header")
            audio_data_header = self.read_next_global_frame_body_extended_data_header()

        if audio_data_header is None:
            print("Data header missing or format unknown")
            return None

        audio_data_body = self.read_next_audio_data_body(audio_data_header.sample_count,
                                                         audio_data_header.channel_count,
                                                         audio_data_header.sample_size)

        if audio_data_body is None:
            print("Data body missing")
            return None

        return AmmosGlobalFrameBody(audio_data_header, audio_data_body)

    def pcm_for_channel(self, a_channel):
        """
        I return the raw pcm audio data for a given channel.

        :param a_channel: the channel I have to extract
        :type a_channel: int

        :rtype: bytes
        """
        for each in self.container.global_frames:
            print(each.global_frame_body)
        return b"".join([each.global_frame_body.data_body.pcm_for_channel(a_channel) for each in self.container.global_frames])
