import json
from flowtimer.Phase import Phase
from flowtimer.RecurringPhaseSequence import RecurringPhaseSequence

"""
I represent a Schedule consisting of blocks. Blocks can be a single phase or
a sequence of phases.
There are 6 states a Schedule can be in: initial, running, completed, aborted, paused, terminated.
"""


class Schedule:

    def __init__(self, title, blocks):
        assert blocks is not []
        self.title = title
        self.blocks = blocks
        self.current_block = blocks[0]
        self.state = "initial"

    @classmethod
    def default_json_string(cls):
        return json.dumps(
            {"title": "Default",
                "blocks": [
                    {"type": "Phase",
                     "title": "MorningHuddle",
                     "initial_ticks": 900
                     },
                    {"type": "Sequence",
                     "title": "AM",
                     "sequence": [
                         {
                             "title": "Tasking",
                             "initial_ticks": 120
                         },
                         {
                             "title": "Working",
                             "initial_ticks": 5400
                         },
                         {
                             "title": "Syncing",
                             "initial_ticks": 300
                         },
                         {
                             "title": "Break",
                             "initial_ticks": 600
                         }
                     ],
                     "initial_repetitions": 2
                     }
                ]
             }
        )

    @classmethod
    def from_json(cls, a_json_string):
        def custom_object_hook(d):
            if 'title' in d and 'blocks' in d:
                return Schedule(d['title'], d['blocks'])
            if 'title' in d and 'initial_ticks' in d:
                return Phase(d['title'], d['initial_ticks'])
            if 'sequence' in d and 'initial_repetitions' in d:
                return RecurringPhaseSequence(d["title"], d['sequence'], d['initial_repetitions'])
            print("Wrong format")
            return d
        return json.loads(a_json_string, object_hook=custom_object_hook)

    @classmethod
    def default(cls):
        return cls.from_json(cls.default_json_string())

    def to_json(self):
        def serialize_block(block):
            if not block.is_sequence():
                return {"type": "Phase", "title": block.title, "initial_ticks": block.initial_ticks}
            if block.is_sequence():
                return {
                    "type": "Sequence",
                    "title": block.title,
                    "sequence": [serialize_block(phase) for phase in block.phases],
                    "initial_repetitions": block.initial_repetitions
                }
            raise TypeError(f"Unknown block type: {type(block)}")

        return json.dumps({
            "title": self.title,
            "blocks": [serialize_block(block) for block in self.blocks]
        })

    def start(self):
        self.state = "running"
        self.current_block.start()

    def pause(self):
        self.state = "paused"

    def running(self):
        return self.state == "running"

    def paused(self):
        return self.state == "paused"

    def abort(self):
        self.current_block.abort()
        self.state = "aborted"

    def current_block_is_final(self):
        index = self.blocks.index(self.current_block)
        return index == (len(self.blocks) - 1)

    def completed(self):
        if (self.current_block.completed()) and (self.current_block_is_final()):
            self.state = "completed"
            return True
        else:
            return False

    def aborted(self):
        return self.state == "aborted"

    def terminated(self):
        return (self.aborted() or self.completed())

    def advance_to_next_block(self):
        """Helper method to move to the next block in the schedule."""
        current_block_index = self.blocks.index(self.current_block)
        if current_block_index < len(self.blocks) - 1:
            self.current_block = self.blocks[current_block_index+1]

    def skip(self):
        if self.current_block_is_final():
            print("Time over")
            self.abort()
            return
        if self.current_block.is_sequence():
            print("Skip the next phase in sequence")
            self.current_block.skip()
            if self.current_block.completed():
                self.current_block = self.upcoming_blocks()[0]
            return
        index = self.blocks.index(self.current_block)
        self.current_block = self.blocks[index+1]
        return

    def total_ticks_left(self):
        return (self.current_block.ticks_left +
                sum([block.ticks_left for block in self.upcoming_blocks()]))

    def upcoming_blocks(self):
        index = self.blocks.index(self.current_block)
        if index < len(self.blocks):
            return self.blocks[index+1:]
        return []

    def tick(self, ticks):
        if not self.completed():
            self.current_block.tick(ticks)
            if self.current_block.completed():
                if self.current_block_is_final():
                    self.state = "completed"
                else:
                    self.skip()
                    return True
