import numpy as np
from datetime import datetime


class PPDWContainer():
    """
    I store multiple signals imported from one or more .ppdw files
    .. automethod:: __init__
    """

    def __init__(self, name, signals=None):

        if signals is None:
            self.signals = []
        else:
            self.signals = signals

    def __str__(self):
        return "\n".join(["Number of pulses:" + str(len(self.signals)),
                          "Start time:" + str(self.start_time()),
                          "End time:" + str(self.end_time())])

    def add(self, a_pdw):
        self.signals.append(a_pdw)

    def as_pulse_dict(self):
        pulse_dict = {}
        pulse_dict[0] = self.signals[0].to_json()
        for index, current_pdw in enumerate(self.signals[1:]):
            pulse_dict[index+1] = current_pdw.to_json()
            pulse_dict[index]["TIMEDELTA"] = current_pdw.time_of_arrival - pulse_dict[index]["TIMEOFARRIVAL"]
        pulse_dict[index+1]["TIMEDELTA"] = 0
        return pulse_dict

    def julian_date_string(self):
        ts = (self.start_time() - np.datetime64('1970-01-01T00:00:00')) / np.timedelta64(1, 's')
        time_tuple = datetime.utcfromtimestamp(ts).timetuple()
        return str(time_tuple.tm_year)[2:] + str(time_tuple.tm_yday).zfill(3)

    def center_frequencies(self):
        return list({each.center_frequency for each in self.signals})

    def channels(self):
        return list({each.channel for each in self.signals})

    def modulations(self):
        return list({each.modulation for each in self.signals})

    def bandwidths(self):
        return list({each.frequency_shift_or_bandwidth for each in self.signals})

    def start_time(self):
        return min([each.time_of_arrival for each in self.signals])

    def end_time(self):
        return max([each.time_of_arrival for each in self.signals])

    def to_json(self):
        return {'JULIANDATE': self.julian_date_string(),
                'STARTTIME': str(self.start_time()),
                'ENDTIME': str(self.end_time()),
                # 'CENTERFREQUENCIES': self.center_frequencies(),
                'CHANNELS': self.channels()
                # 'MODULATIONS': self.modulations(),
                # 'BANDWIDTHS': self.bandwidths()
                }

if __name__ == '__main__':
    pass
