import tkinter as tk
import tkinter.ttk as ttk
from tkinter import filedialog, messagebox
from PIL import Image, ImageTk
from pathlib import Path
import argparse
import json
from Schedule import Schedule


class TimerApp(tk.Frame):

    def __init__(self, parent, tick_speed, config_file, color_file, autostart):

        super().__init__(parent)
        self.parent = parent
        self.tick_speed = tick_speed
        self.photo = ImageTk.PhotoImage(file=Path(__file__).parent /
                                        'resources' / 'flowtimer_startbg_new.png')
        print("Config:", config_file)
        self.schedule = self.load_config(config_file)
        self.color_scheme = self.load_color_file(color_file)
        self.build_gui()
        if autostart:
            self.toggle_play_pause()

    def build_gui(self):

        s = ttk.Style()
        s.theme_use('clam')
        s.configure("orange.Horizontal.TProgressbar", troughcolor='ivory3',
                    bordercolor='black', background='CadetBlue1',
                    lightcolor='white', darkcolor='black')

        self.headline_frame = tk.Frame(self.parent, bg='white')
        self.headline_frame.pack(side='top', fill='both', expand=False)
        self.center_frame = tk.Frame(self.parent, bg='black')
        self.center_frame.pack(side='top', fill='both', expand=True)
        self.button_frame = tk.Frame(root)
        self.button_frame.pack(side='bottom', fill='both', expand=False)

        self.label_headline = tk.Label(self.headline_frame,
                                       text=(self.schedule.current_block.title),
                                       bg='white', fg='grey', font="serif 20")
        self.label_headline.pack(side='left', fill='both', expand=True)

        self.label_config = tk.LabelFrame(self.headline_frame,
                                          text=f"config: {self.schedule.title}",
                                          bg='white', font="serif 12")
        self.label_config.pack(side='right', fill='both', expand=False,
                               ipadx=20, padx=20, pady=10)

        self.config_button = tk.Button(self.headline_frame, text="change config",
                                       font="courier 14", width=20, command=self.select_config)
        self.config_button.pack(side='right', padx=10, pady=10, expand=False)

        self.label_config_text = tk.Label(self.label_config, text=self.current_config(),
                                          bg='white', font="serif 10", justify='right')
        self.label_config_text.pack()

        self.label_start = tk.Label(self.center_frame, image=self.photo, bg='black')
        self.label_start.pack(side='left', fill='both', expand=True)

        self.label_sequence = tk.Label(self.center_frame, bg='red')
        self.label_sequence.pack(side='top', fill='both', expand=True)

        self.progressbar = ttk.Progressbar(self.center_frame,
                                           style="orange.Horizontal.TProgressbar",
                                           orient="horizontal", length=600, mode="determinate")
        self.progressbar["maximum"] = 59
        self.progressbar.pack(side='top')
        self.label_duration = tk.Label(self.center_frame, bg='green')
        self.label_duration.pack(side='top', fill='both', expand=True)

        self.freeze_button = tk.Button(self.button_frame, text="PLAY/PAUSE",
                                       font="courier 14", width=20, command=self.toggle_play_pause)
        self.freeze_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.skip_button = tk.Button(self.button_frame, text=("SKIP"),
                                     font="courier 14", width=20, command=self.skip)
        self.skip_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.quit_button = tk.Button(self.button_frame, text="QUIT",
                                     font="courier 14", width=20, command=self.quit_app)
        self.quit_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.parent.resizable(False, False)

    def ticks_left_in_phase(self):
        if self.schedule.current_block.is_sequence():
            return self.schedule.current_block.current_phase.ticks_left
        else:
            return self.schedule.current_block.ticks_left

    def colors_for_phase(self, a_title):
        return next((d for d in self.color_scheme if d.get('title') == a_title), None)

    def current_title(self):
        if self.schedule.current_block.is_sequence():
            return self.schedule.current_block.current_phase.title
        else:
            return self.schedule.current_block.title

    def current_config(self):
        if self.schedule.current_block.is_sequence():
            return (f"{self.schedule.current_block.title} / "
                    f"{self.schedule.current_block.current_phase.title} / "
                    f"{self.schedule.current_block.current_phase_number()+1} of "
                    f"{len(self.schedule.current_block.phases)} "
                    f"({self.schedule.current_block.passes_left})")
        return (f"{self.schedule.current_block.title}")

    def current_time_status(self):
        if self.ticks_left_in_phase() < 60:
            return f"{self.ticks_left_in_phase()} seconds left in phase {self.current_title()}"
        else:
            return f"{(self.ticks_left_in_phase() // 60)} min left in phase {self.current_title()}"

    # FIXME: This logic looks like shit. Refactor it!!!
    def tick(self):
        current_process = self.after(1000, self.tick)
        if self.schedule.is_terminated():
            self.label_sequence.config(text=("\n" + "\nTime over !"),
                                       bg=self.colors_for_phase("Completed").get('sequence_label'),
                                       fg="white")
            self.label_duration.config(text="", bg=self.colors_for_phase("Completed").get('duration_label'),
                                       fg="white")
            self.progressbar.pack_forget()
            self.center_frame.configure(bg=self.colors_for_phase("Completed").get('center_frame'))
            self.after_cancel(current_process)
            self.skip_button['state'] = 'disabled'
            self.skip_button.config(fg="ivory3")
            self.freeze_button['state'] = 'disabled'
            self.freeze_button.config(fg="ivory3")
            return
        if self.schedule.is_running():
            self.schedule.tick(1*self.tick_speed)
            self.progressbar["value"] = (self.schedule.current_block.initial_ticks -
                                         self.schedule.current_block.ticks_left) % 60
            self.progressbar.update()
            self.label_duration.config(text=self.current_time_status(), font='times 22')
            self.label_config_text.config(text=self.current_config())

    def skip(self):
        self.schedule.skip()

    def toggle_play_pause(self):
        if self.schedule.is_initial():
            self.label_start.pack_forget()
            self.schedule.start()
            self.tick()
            return
        if not self.schedule.is_running():
            self.freeze_button.config(relief="raised", fg='black')
            self.label_sequence.config(fg="white")
            self.label_duration.config(fg="white")
            self.schedule.start()
            self.tick()
        else:
            self.schedule.pause()
            self.freeze_button.config(relief="sunken", fg="red")
            self.label_sequence.config(text=("timer paused !"), fg="red", bg="black",
                                       font='times 72')
            self.label_duration.config(text="", bg="black")
            self.center_frame.config(bg="black")

    def quit_app(self):
        self.parent.destroy()

    def load_color_file(self, file_name=Path(__file__).parent / 'configs' / 'colors.json'):
        try:
            with open(file_name) as color_file:
                colors = json.loads(color_file.read())
        except Exception as e:
            with open(Path(__file__).parent / 'configs' / 'colors.json') as color_file:
                colors = json.loads(color_file.read())
            print(e)
        return colors.get('color_list', [])

    def change_config(self, json_string):
        self.schedule = Schedule.from_json(json_string)
        return self.schedule

    def load_config(self, file_name=Path(__file__).parent / 'configs' / 'default.json'):
        try:
            with open(file_name) as config_file:
                schedule = self.change_config(config_file.read())
        except Exception as e:
            messagebox.showerror('Config error',
                                 f'Can not load {file_name}\n\nDefault loaded instead')
            print(e)
            return Schedule.default()
        return schedule

    def select_config(self):
        self.config_files = [("all files", "*.json")]
        self.answer = filedialog.askopenfilename(parent=root,
                                                 initialdir=Path(__file__).parent / 'configs',
                                                 title="Please choose a config file:",
                                                 filetypes=self.config_files)
        self.load_config(Path(self.answer))


parser = argparse.ArgumentParser(description="Load specified config file and autostart if needed")
parser.add_argument('--start', dest='autostart', action='store_true', default=False,
                    required=False, help='use --start to autostart the timer with the given config')
parser.add_argument('--config', dest='config_file', required=False,
                    default=Path(__file__).parent / 'configs' / 'default.json',
                    help='use --config /path/to/config_file.json to specify the configuration of the timer')
parser.add_argument('--colors', dest='color_file', required=False,
                    default=Path(__file__).parent / 'configs' / 'colors.json')
args = parser.parse_args()

root = tk.Tk()
root.title(args.config_file)
root.geometry("1280x860")
root.config(bg='black')
root.iconphoto(False, ImageTk.PhotoImage(Image.open(Path(__file__).parent /
                                                    'resources' /
                                                    'icon_bombtimer.png')))

app = TimerApp(root,
               tick_speed=1,
               config_file=args.config_file,
               color_file=args.color_file,
               autostart=args.autostart)

app.mainloop()
