"""I provide an AMMOS data header for audio data frames."""

import struct
from ammosreader.AmmosConstants import AmmosAudioDemodType


class AmmosAudioDataHeader:
    """I implement an AMMOS data header for audio data frames."""

    HEADER_SIZE = 36  # 9 words

    @classmethod
    def from_bytes(cls, in_bytes):
        """I return an AMMOS data header from given bytes."""
        assert len(in_bytes) == cls.HEADER_SIZE
        elements = struct.unpack('<IIQIIIII', in_bytes)
        sample_rate = elements[0]
        status = elements[1]
        frequency = elements[2]
        demod_bandwidth = elements[3]
        demod_type = AmmosAudioDemodType(elements[4])
        number_of_samples = elements[5]
        number_of_channels = elements[6]
        sample_size = elements[7]
        return AmmosAudioDataHeader(sample_rate, status, frequency, demod_bandwidth, demod_type,
                                    number_of_samples, number_of_channels, sample_size)

    def __init__(self, sample_rate, status, frequency, demod_bandwidth, demod_type, number_of_samples,
                 number_of_channels, sample_size):
        """I create a new instance of myself using the above parameters."""
        self.sample_rate = sample_rate
        self.status = status
        self.frequency = frequency
        self.demod_bandwidth = demod_bandwidth
        self.demod_type = AmmosAudioDemodType(demod_type)
        self.number_of_samples = number_of_samples
        self.number_of_channels = number_of_channels
        self.sample_size = sample_size

    def __str__(self):
        """I return the string representation of myself."""
        return ("\nAmmosAudioDataHeader\n" +
                "Sample rate:" + str(self.sample_rate) + "\n" +
                "Status:" + str(self.status) + "\n" +
                "Frequency:" + str(self.frequency) + "\n" +
                "Demodulation bandwidth:" + str(self.demod_bandwidth) + "\n" +
                "Demodulation type:" + str(self.demod_type) + "\n" +
                "Sample count:" + str(self.number_of_samples) + "\n" +
                "Channel count:" + str(self.number_of_channels) + "\n" +
                "Sample size:" + str(self.sample_size) + "\n")
