import struct


class AmmosIFDataBlockHeader:
    """I implement an AMMOS IF data block header."""

    HEADER_SIZE = 4

    @classmethod
    def from_bytes(cls, bytes):
        """I return an AMMOS IF data block header built from bytes."""
        assert len(bytes) == cls.HEADER_SIZE
        elements = struct.unpack('<ccH', bytes)
        header = cls(bytes)
        header.invalidity = ((elements[0] & 1) == 1)
        header.blanking = ((elements[0] & 2) == 1)
        header.user_data = ((elements[0] & 252))
        header.reserved = elements[1]
        header.reciprocal_gain = elements[2]
        return header

    def __init__(self):
        """I return an AMMOS IF data block header initialized with sane default values."""
        self.__invalidity = True
        self.__blanking = False
        self.__user_data = 0
        self.__reserved = 0
        self.__reciprocal_gain = 0

    @property
    def invalidity(self):
        return self.__invalidity

    @invalidity.setter
    def invalidity(self, a_boolean):
        self.__invalidity = a_boolean

    @property
    def blanking(self):
        return self.__blanking

    @blanking.setter
    def blanking(self, a_boolean):
        self.__blanking = a_boolean

    @property
    def user_data(self):
        return self.__user_data

    @user_data.setter
    def user_data(self, some_data):
        self.__user_data = some_data

    @property
    def reserved(self):
        return self.__reserved

    @reserved.setter
    def reserved(self, some_data):
        assert some_data == 0
        self.__reserved = some_data

    @property
    def reciprocal_gain(self):
        return self.__reciprocal_gain

    @reciprocal_gain.setter
    def reciprocal_setter(self, a_gain_value):
        assert 0 <= a_gain_value < pow(2, 16)
        self.__reciprocal_gain = a_gain_value
