import json

"""
I represent a Schedule consisting of blocks. Blocks can be a single phase or
a sequence of phases. 
"""


class Schedule:

    def __init__(self, block_list):
        assert block_list is not []
        self.block_list = block_list
        self.current_block = block_list[0]
        self.state = "initial"

    def to_json(self):
        return json.dumps(self.__dict__)

    def start(self):
        self.state = "running"
        self.current_block.start()

    def pause(self):
        self.state = "paused"

    def running(self):
        return self.state == "running"

    def is_paused(self):
        return self.state == "paused"

    def abort(self):
        self.current_block.abort()
        self.state = "finished"

    def finished(self):
        if (self.current_block.finished()) and (self.block_list[-1] == self.current_block):
            self.state = "finished"
            return True
        else:
            return False

    def skip(self):
        if self.current_block.is_sequence:
            self.current_block.skip()
        else:
            if self.current_block_is_final():
                self.abort()
            else:
                index = self.block_list.index(self.current_block)
                self.current_block = self.block_list[index+1]

    def total_ticks_left(self):
        return (self.current_block.ticks_left +
                sum([block.ticks_left for block in self.upcoming_blocks()]))

    def upcoming_blocks(self):
        index = self.block_list.index(self.current_block)
        if index < len(self.block_list):
            return self.block_list[index+1:]
        return []

    def current_block_is_final(self):
        index = self.block_list.index(self.current_block)
        return index == (len(self.block_list) - 1)

    def tick(self, duration):
        if not self.finished():
            self.current_block.tick(duration)
            if self.current_block.finished():
                if self.current_block_is_final():
                    self.abort()
                else:
                    self.skip()
                    return True
