"""I provide a base class for specialized AmmosReaders."""

from abc import ABC, abstractmethod

from ammosreader.AmmosGlobalFrameHeader import AmmosGlobalFrameHeader
from ammosreader.AmmosSingleFrame import AmmosSingleFrame
from ammosreader.AmmosContainer import AmmosContainer
from ammosreader.AmmosConstants import FrameType
from ammosreader import logger


class AbstractAmmosReader(ABC):
    """I implement a base class for specialized AmmosReaders."""

    def __init__(self, source):
        """
        I am the standard constructor for Ammos Readers.

        Additional information about the file can be added as key/value pairs in tags

        :param source: The source to read Ammos data from
        :type source: AmmosSource
        """
        self.__source = source
        self.__container = AmmosContainer(self.__source.name, [])
        self.__tags = {}

    @property
    def source(self):
        """I return the source of this reader."""
        return self.__source

    @property
    def container(self):
        """I return the container which stores the data read."""
        return self.__container

    @property
    def tags(self):
        """I return all the tags."""
        return self.__tags

    def add_tag(self, a_key, a_value):
        """I add information to tags using a key/value pair."""
        assert a_key not in self.__tags
        self.__tags[a_key] = a_value

    def resync(self):
        """I try to resync, when reading from source was garbled."""
        logger.info("Resync")
        self.source.resync()

    def read_all_frames_left(self):
        """
        I read all remaining frames into my container until end of file is reached.

        :return: a container containing all frames read
        :rtype: AmmosContainer
        """
        frames_read = 0
        while True:
            logger.info("Reading single frame %s ...", frames_read)
            current_frame = self.read_next_single_frame()
            if current_frame is not None:
                frames_read += 1
                self.container.add_frame(current_frame)
            else:
                logger.debug("Frame: %s incomplete", frames_read+1)
                break
        logger.info("%s frames read", len(self.container.global_frames))
        return self.container

    def read_next_global_frame_header(self):
        """
        I return the next global frame header read from current position in file.

        :return: the next global frame header or None if incomplete
        :rtype: AmmosGlobalFrameHeader
        """
        header_size = AmmosGlobalFrameHeader.HEADER_SIZE

        in_bytes = self.__source.read_bytes(header_size)
        logger.info("Reading next global frame header")
        if ((not in_bytes) or (len(in_bytes) < header_size)):
            logger.debug("Can not read all %s bytes of global frame header", header_size)
            return None

        logger.debug("in_bytes %s %s", in_bytes, len(in_bytes))
        current_global_frame_header = AmmosGlobalFrameHeader.from_bytes(in_bytes)

        if current_global_frame_header is None:
            return None

        # logger.info("Current global frame header %s", current_global_frame_header)
        return current_global_frame_header

    def read_bytes(self, bytes_to_read):
        """My descendents have to implement this."""
        return self.__source.read_bytes(bytes_to_read)

    @abstractmethod
    def read_next_global_frame_body(self, data_header_length):
        """My descendents have to implement this."""
        pass

    def read_next_single_frame(self):
        """
        I read and return a single global frame.

        :return: a single global frame
        :rtype: AmmosSingleFrame
        """
        global_frame_header = self.read_next_global_frame_header()

        if global_frame_header is None:
            logger.debug("Global frame header missing")
            return None

        if global_frame_header.data_header_length is None:
            logger.debug("Data header length empty")
            return None

        if global_frame_header.frame_type not in list(FrameType):
            logger.info("Unknown frame type %s found", global_frame_header.frame_type)
            return None

        # FIXME: Refactor duplicate logic - This code stinks

        global_frame_body = None

        if_data_types = [FrameType.IF_DATA_32BIT_REAL_IMAGINARY_FIXEDPOINT.value,
                         FrameType.IF_DATA_16BIT_REAL_IMAGINARY_FIXEDPOINT.value,
                         FrameType.IF_DATA_16BIT_REAL_REAL.value,
                         FrameType.IF_DATA_32BIT_REAL_IMAGINARY_FIXEDPOINT_RESCALED.value,
                         FrameType.IF_DATA_32BIT_REAL_IMAGINARY_FLOATINGPOINT_RESCALED.value]

        if global_frame_header.frame_type == FrameType.AUDIO_DATA.value:
            logger.info("Audio Datastream found")
            global_frame_body = self.read_next_global_frame_body(global_frame_header.data_header_length)
            if global_frame_body is None:
                logger.info("Can not read global frame body for audio data stream")
                return None
            return AmmosSingleFrame(global_frame_header, global_frame_body)

        if global_frame_header.frame_type in if_data_types:
            logger.info("IF Datastream found")
            global_frame_body = self.read_next_global_frame_body(global_frame_header.data_header_length)
            if global_frame_body is None:
                logger.info("Can not read global frame body for IF data stream")
                return None
            return AmmosSingleFrame(global_frame_header, global_frame_body)

        if global_frame_body is None:
            logger.info("Can not read global frame body")
            return None

        logger.info("Unsupported frame type %s found", global_frame_header.frame_type)
        return None
