import tkinter as tk
import tkinter.ttk as ttk
from tkinter import filedialog
from Schedule import Schedule
from PIL import Image, ImageTk
from pathlib import Path
import datetime


class TimerApp(tk.Frame):

    def __init__(self, parent, tickspeed):

        super().__init__(parent)
        self.parent = parent
        self.tick_speed = tickspeed
        self.count = 0
        self.load_config()
        self.photo = ImageTk.PhotoImage(file=Path(__file__).parent /
                                        'resources' / 'flowtimer_startbg_new.png')
        self.config_state = 'default'

        self.build_gui()

    def build_gui(self):

        s = ttk.Style()
        s.theme_use('clam')
        s.configure("orange.Horizontal.TProgressbar", troughcolor='ivory3',
                    bordercolor='black', background='CadetBlue1',
                    lightcolor='white', darkcolor='black')

        self.headline_frame = tk.Frame(self.parent, bg='white')
        self.headline_frame.pack(side='top', fill='both', expand=False)
        self.center_frame = tk.Frame(self.parent, bg='black')
        self.center_frame.pack(side='top', fill='both', expand=True)
        self.button_frame = tk.Frame(root)
        self.button_frame.pack(side='bottom', fill='both', expand=False)

        self.label_headline = tk.Label(self.headline_frame,
                                       text=("Hochintensive Intervallarbeit"),
                                       bg='white', fg='grey', font="serif 20")
        self.label_headline.pack(side='left', fill='both', expand=True)

        self.label_config = tk.LabelFrame(self.headline_frame, text="config: ",
                                          bg='white', font="serif 12")
        self.label_config.pack(side='right', fill='both', expand=False,
                               ipadx=20, padx=20, pady=10)

        self.config_button = tk.Button(self.headline_frame, text="change config",
                                       font="courier 14", width=20, command=self.select_config)
        self.config_button.pack(side='right', padx=10, pady=10, expand=False)

        self.label_config_text = tk.Label(self.label_config, text=self.current_config(),
                                          bg='white', font="serif 10", justify='right')
        self.label_config_text.pack()

        self.label_start = tk.Label(self.center_frame, image=self.photo, bg='black')
        self.label_start.pack(side='left', fill='both', expand=True)

        self.label_sequence = tk.Label(self.center_frame, bg='red')
        self.label_sequence.pack(side='top', fill='both', expand=True)

        self.progressbar = ttk.Progressbar(self.center_frame,
                                           style="orange.Horizontal.TProgressbar",
                                           orient="horizontal", length=600, mode="determinate")
        self.progressbar["maximum"] = 59
        self.progressbar.pack(side='top')
        self.label_duration = tk.Label(self.center_frame, bg='green')
        self.label_duration.pack(side='top', fill='both', expand=True)

        self.start_timer_button = tk.Button(self.button_frame, text="START",
                                            font="courier 14", width=20, command=self.start)
        self.start_timer_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.freeze_button = tk.Button(self.button_frame, text="PLAY/PAUSE",
                                       font="courier 14", width=20, command=self.toggle_tick)
        self.freeze_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.skip_button = tk.Button(self.button_frame, text=("SKIP"),
                                     font="courier 14", width=20, command=self.skip)
        self.skip_button.pack(side='left', fill='both', ipady=20, expand=True)

        self.quit_button = tk.Button(self.button_frame, text="QUIT",
                                     font="courier 14", width=20, command=self.quit_app)
        self.quit_button.pack(side='left', fill='both', ipady=20, expand=True)

    def ticks_left_in_phase(self):
        if self.schedule.current_block.is_sequence():
            return self.schedule.current_block.current_phase.ticks_left
        else:
            return self.schedule.current_block.ticks_left

    def ticks_left_in_block(self):
        if self.schedule.current_block.is_sequence():
            return self.schedule.current_block.ticks_left
        else:
            return self.schedule.current_block.ticks_left

    def current_title(self):
        if self.schedule.current_block.is_sequence():
            return self.schedule.current_block.current_phase.title
        else:
            return self.schedule.current_block.title

    def current_config(self):
        if self.schedule.current_block.is_sequence():
            return f"{self.schedule.current_block.title} / {self.schedule.current_block.current_phase.title} / {self.schedule.current_block.current_phase_number()+1} of {len(self.schedule.current_block.phases)} ({self.schedule.current_block.passes_left})"

    def current_time_status(self):
        if self.ticks_left_in_phase() < 60:
            return f"{self.ticks_left_in_phase()} seconds left in phase {self.current_title()}"
        else:
            return f"{(self.ticks_left_in_phase() // 60)} min left in phase {self.current_title()}"

    def tick(self):

        if not self.schedule.running():
            return
        self.label_start.pack_forget()
        current_process = self.after(1000, self.tick)
        if self.schedule.state == 'initial':
            self.label_sequence.config(text=("\n" + str(self.schedule.current_block.title) + "..."))
            self.label_duration.config(text=self.current_time_status())
            self.schedule.start()
            self.schedule.tick(1*self.tick_speed)
        else:
            if self.schedule.running():
                self.schedule.tick(1*self.tick_speed)
                self.progressbar["value"] = (self.schedule.current_block.initial_ticks -
                                             self.schedule.current_block.ticks_left) % 60
                self.progressbar.update()
                self.label_duration.config(text=self.current_time_status())
                self.label_config_text.config(text=self.current_config())
            else:
                self.label_sequence.config(text=("\n" + "\nTime over !"), bg="red", fg="white")
                self.label_duration.config(text="", bg="red", fg="white")
                self.progressbar.pack_forget()
                self.center_frame.configure(bg="red")
                self.after_cancel(current_process)
                self.skip_button['state'] = 'disabled'
                self.skip_button.config(fg="ivory3")
                self.freeze_button['state'] = 'disabled'
                self.freeze_button.config(fg="ivory3")

    def start(self):
        self.schedule.start()
        self.tick()
        self.start_timer_button['state'] = 'disabled'
        self.start_timer_button.config(fg="ivory3")
        self.config_button.pack_forget()

    def skip(self):
        self.schedule.skip()

    def toggle_tick(self):
        if self.schedule.paused():
            self.freeze_button.config(relief="raised", fg='black')
            self.label_sequence.config(fg="white")
            self.label_duration.config(fg="white")
            self.schedule.start()
            self.tick()
        else:
            self.schedule.pause()
            self.freeze_button.config(relief="sunken", fg="red")
            self.label_sequence.config(text=("\n" + "\n! timer paused !"), fg="red", bg="black")
            self.label_duration.config(text="", bg="black")
            self.center_frame.config(bg="black")

    def quit_app(self):
        self.parent.destroy()

    def change_config(self, json_string):
        self.schedule = Schedule.from_json(json_string)

    def load_config(self, file_name=Path(__file__).parent / 'configs' / 'default.json'):
        with open(file_name) as config_file:
            self.change_config(config_file.read())

    def select_config(self):
        self.config_state = 'user'
        self.config_files = [("all files", "*.json")]
        self.answer = filedialog.askopenfilename(parent=root,
                                                 initialdir=Path(__file__).parent / 'configs',
                                                 title="Please choose a config file:",
                                                 filetypes=self.config_files)
        self.load_config(Path(self.answer))


now = datetime.datetime.now()
date = now.strftime("%m/%d/%Y")
time = now.strftime("%H:%M:%S")

root = tk.Tk()
root.title("--=> flowtimer <=-- " + "  date: " + date + "  time: " + time)
root.geometry("1280x860")
root.config(bg='black')
root.iconphoto(False, ImageTk.PhotoImage(Image.open(Path(__file__).parent / 'resources' / 'icon_bombtimer.png')))

app = TimerApp(root, tickspeed=1)

app.mainloop()
