"""I provide an Ammos container."""


class AmmosContainer():
    """I store the content of an R&S Ammos file in a more accessible way."""
    def __init__(self, name, frames):
        self.__name = name
        self.__global_frames = frames
        self.__tags = []

    @property
    def name(self):
        return self.__name

    @name.setter
    def name(self, a_name):
        self.__name = a_name

    @property
    def global_frames(self):
        return self.__global_frames

    def add_tag(self, tag):
        self.__tags.append(tag)

    def add_frame(self, frame):
        self.__global_frames.append(frame)

    def size(self):
        # return sum([each.global_frame_header.frame_length for each in self.__global_frames])
        return sum(each.global_frame_header.frame_length for each in self.__global_frames)

    def frequencies(self):
        """I return a list of unique frequencies inside this container."""
        return list({each.global_frame_body.data_header.frequency for each in self.__global_frames})

    def frame_types(self):
        return list({each.global_frame_header.frame_type for each in self.__global_frames})

    def frame_sizes(self):
        return [each.global_frame_header.frame_length for each in self.__global_frames]

    def unique_frame_sizes(self):
        return list({each.global_frame_header.frame_length for each in self.__global_frames})

    def is_homogenic(self):
        return (len(self.unique_frame_sizes()) == 1) and (len(self.frame_types()) == 1)

    def __str__(self):
        start_time = self.__global_frames[0].global_frame_body.data_header.timestamp
        end_time = self.__global_frames[-1].global_frame_body.data_header.timestamp

        frq = str(self.__global_frames[0].global_frame_body.data_header.frequency)

        return ("Start time: " + str(start_time) +
                "\nEnd time  : " + str(end_time) + "\nFrequencies: " + frq)

    def payload(self):
        return [each.global_frame_body.data_body.data for each in self.__global_frames]

    def pcm_for_channel(self, a_channel):
        """
        I return the raw pcm audio data for a given channel.

        :param a_channel: the channel I have to extract
        :type a_channel: int

        :rtype: bytes
        """
        return (b"".join([each.global_frame_body.data_body.pcm_for_channel(a_channel)
                          for each in self.global_frames]))
