import tkinter as tk
from tkinter import ttk, messagebox
from flexoentity import EntityState
from builder.exam_elements import AnswerOption
from .answer_options_dialog import AnswerOptionsDialog
from .attach_media_dialog import AttachMediaDialog

class OptionQuestionEditorDialog(tk.Toplevel):
    """Dialog for editing OptionQuestions (SingleChoiceQuestion, MultipleChoiceQuestion)."""

    def __init__(self, parent, question=None, available_domains=None):
        super().__init__(parent)
        self.title("Edit Options")
        self.transient(parent)
        # ensure window is displayed before grabbing
        self.update_idletasks()
        self.wait_visibility()
        self.grab_set()
        self.result = None

        self.question = question
        self.available_domains = sorted(available_domains or [])
        self.domain_var = tk.StringVar(value=self.question.domain_code if self.question else "")
        self.state_var = tk.StringVar(value=self.question.state.name if self.question else EntityState.DRAFT.name)

        self.title(f"Edit Option Question {getattr(self.question, 'flexo_id', '')}")
        self.create_widgets()
        self.populate_fields()

    def create_widgets(self):
        frm = ttk.Frame(self, padding=10)
        frm.pack(fill="both", expand=True)

        # Domain
        ttk.Label(frm, text="Domain").pack(anchor="w")
        self.cmb_domain = ttk.Combobox(frm, textvariable=self.domain_var, values=self.available_domains, state="readonly")
        self.cmb_domain.pack(fill="x", pady=(0, 10))

        # Question text
        ttk.Label(frm, text="Question Text").pack(anchor="w")
        self.txt_question = tk.Text(frm, height=5, wrap="word")
        self.txt_question.pack(fill="x", pady=(0, 10))

        # Status / State
        ttk.Label(frm, text="Status").pack(anchor="w")
        self.cmb_status = ttk.Combobox(frm, textvariable=self.state_var, values=self.question.allowed_transitions(), state="readonly")
        self.cmb_status.pack(fill="x", pady=(0, 10))

        # Answers section
        ttk.Label(frm, text="Answers").pack(anchor="w")
        self.btn_edit_answers = ttk.Button(frm, text="Edit Answers…", command=self.open_answer_editor)
        self.btn_edit_answers.pack(anchor="w", pady=(5, 10))

        ttk.Button(frm, text="Attach Media...", command=self.open_media_dialog).pack(pady=5)


        # Bottom buttons
        btn_frame = ttk.Frame(frm)
        btn_frame.pack(fill="x", pady=(10, 0))
        ttk.Button(btn_frame, text="OK", command=self.on_ok).pack(side="right", padx=5)
        ttk.Button(btn_frame, text="Cancel", command=self.on_cancel).pack(side="right")

    def populate_fields(self):
        self.txt_question.delete("1.0", tk.END)
        text_value = getattr(self.question, "text", "")
        self.txt_question.insert("1.0", text_value or "")

    def open_answer_editor(self):
        """Launch the unified AnswerOptionsDialog."""
        dlg = AnswerOptionsDialog(self, options=self.question.options)
        if dlg.result:
            self.question.options = [
                AnswerOption(o["id"], o["text"], o["points"]) for o in dlg.result
            ]
            messagebox.showinfo("Answers updated", f"{len(dlg.result)} answers saved.")

    def open_media_dialog(self):
        AttachMediaDialog(self, self.question)

    def on_ok(self):
        # FIXME: Use from dict here
        text = self.txt_question.get("1.0", tk.END).strip()
        if not text:
            messagebox.showwarning("Empty", "Question text cannot be empty.")
            return

        self.question.text = text
        self.question.domain_code = self.cmb_domain.get()
        new_state = EntityState[self.cmb_status.get()]
        self.question.apply_state_change(new_state)
        self.result = self.question
        self.destroy()

    def on_cancel(self):
        self.destroy()
