import tkinter as tk
from tkinter import filedialog, messagebox, ttk, font, simpledialog
import json
import copy
from pathlib import Path
from datetime import datetime
from flexoentity import FlexOID
from builder.exam import Exam
from builder.question_catalog import QuestionCatalog
from builder.media_items import NullMediaItem
from builder.catalog_manager import CatalogManager
from builder.exam_manager import ExamManager
from builder.exam_elements import SingleChoiceQuestion, MultipleChoiceQuestion, InstructionBlock
from .menu import AppMenu
from .actions_panel import ActionsPanel
from .select_panel import SelectPanel
from .detail_panel import DetailPanel
from .option_question_editor import OptionQuestionEditorDialog
from .default_question_editor import DefaultQuestionEditorDialog
from .help_dialog import OrgHelpDialog
from .create_exam_dialog import ExamDialog
from .exam_layout_editor import ExamLayoutDialog
from .session_manager import SessionManager

class ExamBuilderApp(tk.Tk):
    def __init__(self):
        super().__init__()
        self.catalog_manager = CatalogManager()
        self.exam_manager = ExamManager()
        self.title("flex-o-grader")

        self.geometry("1000x600")
        default_font = font.nametofont("TkDefaultFont")
        default_font.configure(size=12)  # increase from default (usually 9–10)

        # Apply same size to other common named fonts
        for name in ("TkTextFont", "TkHeadingFont", "TkMenuFont"):
            try:
                font.nametofont(name).configure(size=12)
            except tk.TclError:
                pass

        self.status_var = tk.StringVar(value="No catalog loaded")
        self.recent_actions_var = tk.StringVar(value="")  # new right-side text
        self.current_qtype_var = tk.StringVar(value="single_choice")
        self.target_exam_var = tk.StringVar(value="")

        self._recent_actions = []
        self.domain_set = set()
        self.clipboard = []  # holds copied Question objects
        self.clipboard_action = None  # "copy"
        self.clipboard_source_catalog = None
        self.config(menu=AppMenu(self))
        self.create_widgets()

        self.log_action("GUI started")

        self.session = SessionManager(catalog_manager=self.catalog_manager,
                                      exam_manager=self.exam_manager)
        geometry = self.session.load()
        self.geometry(geometry)
        self.after(60000, lambda: self.session.save(self.geometry()))
        self.protocol("WM_DELETE_WINDOW", self.on_quit)
        self.refresh_all()

    def refresh_all(self):
        self.refresh_catalog_list()
        self.refresh_exam_list()
        self.refresh_question_tree()
        self.refresh_status_bar()
        if self.question_tree.selection():
            self.detail_panel.refresh_details(self.require_selected_question(),
                                              self.require_active_catalog())
    @property
    def active_catalog(self):
        """Return the currently active catalog, or None if none is set."""
        return self.catalog_manager.get_active()

    def require_active_catalog(self):
        """Return the active catalog, or show a message and return None."""
        active = self.active_catalog
        if not active:
            messagebox.showinfo("No Catalog", "Please create or open a catalog first.")
        return active

    def create_widgets(self):
        self.columnconfigure(0, weight=2, uniform="columns")
        self.columnconfigure(1, weight=1, uniform="columns")
        self.rowconfigure(0, weight=1)

        self.selection_panel = SelectPanel(self)

        self.action_panel = ActionsPanel(self)

        self.detail_panel = DetailPanel(self)
        # --- Status bar ---

        status_frame = ttk.Frame(self, relief="sunken", padding=(6, 3))
        status_frame.grid(row=1, column=0, columnspan=2, sticky="ew")

        status_frame.columnconfigure(0, weight=1)
        status_frame.columnconfigure(1, weight=0)

        # Unified style for both labels
        status_style = ttk.Style()
        status_style.configure("Status.TLabel", anchor="w", font=("Segoe UI", 10))

        ttk.Label(status_frame, textvariable=self.status_var, style="Status.TLabel").grid(
            row=0, column=0, sticky="w"
        )
        ttk.Label(status_frame, textvariable=self.recent_actions_var, style="Status.TLabel",
                  anchor="e", justify="right").grid(
                      row=0, column=1, sticky="e", padx=(20, 4)
                  )

        menu_font = font.nametofont("TkMenuFont")
        menu_font.configure(family="Segoe UI", size=12)

    def show_help(self):
        OrgHelpDialog(self, "help/README.org")

    def log_action(self, text: str, max_actions: int = 3):
        """Keep track of the last few user actions for display in the status bar."""
        self._recent_actions.append(text)
        self._recent_actions = self._recent_actions[-max_actions:]
        self.recent_actions_var.set(" | ".join(self._recent_actions))

    def export_catalog(self):
        """Export the currently active catalog to a JSON file."""
        catalog = self.catalog_manager.get_active()
        if not catalog:
            messagebox.showwarning("No Catalog", "No active catalog to export.")
            return

        # Ask for a file location
        file_path = filedialog.asksaveasfilename(
            defaultextension=".json",
            file_types=[("JSON files", "*.json"), ("All files", "*.*")],
            title="Export Catalog As"
        )
        if not file_path:
            return  # user cancelled

        try:
            # Get catalog data as dict or JSON string
            data = catalog.to_dict() if hasattr(catalog, "to_dict") else catalog
            Path(file_path).write_text(json.dumps(data, indent=2, ensure_ascii=False))
            messagebox.showinfo("Export Successful", f"Catalog exported to:\n{file_path}")
        except Exception as e:
            messagebox.showerror("Export Failed", f"Error exporting catalog:\n{e}")

    def open_catalog(self):
        path = filedialog.askopenfilename(filetypes=[("Catalog JSON", "*.json")])
        if not path:
            return

        with open(path, "r", encoding="utf-8") as f:
            data = json.load(f)

        catalog = QuestionCatalog.from_dict(data)
        title = simpledialog.askstring("New Catalog", "Enter title:", parent=self)
        catalog.title = title
        self.catalog_manager.add_catalog(catalog)
        self.catalog_manager.set_active(catalog.flexo_id)
        self.refresh_all()

    def refresh_catalog_list(self):
        self.catalog_dropdown["values"] = self.catalog_manager.list_titles()
        self.catalog_var.set(self.catalog_manager.get_active_display_title())

    def refresh_status_bar(self):
        current_catalog = self.active_catalog
        if not current_catalog:
            self.status_var.set("No catalog loaded")
            return
        q_count = len(current_catalog.questions)
        title = current_catalog.title
        version = current_catalog.version
        status = (
            f"Catalog: {title} | Version: {version} | Questions: {q_count} | "
            f"Status: {current_catalog.status_text}"
        )
        self.status_var.set(status)

    @property
    def questions(self):
        if not self.active_catalog:
            return []
        # Always provide deterministic order (e.g. by question ID)
        return sorted(self.active_catalog.questions, key=lambda q: q.id)

    def create_catalog(self):
        did = simpledialog.askstring("New Catalog", "Enter domain:", parent=self)
        if not did:
            return
        title = simpledialog.askstring("Catalog Title", "Enter title:", parent=self)
        if not title:
            return
        author = (
            simpledialog.askstring("Author", "Enter author:", parent=self) or "unknown"
        )
        catalog = QuestionCatalog.with_domain(did)
        catalog.title = title
        catalog.author =author
        catalog._update_fingerprint()
        self.catalog_manager.add_catalog(catalog)
        self.catalog_manager.set_active(did)

        # Clear GUI state for new catalog

        self.question_tree.delete(*self.question_tree.get_children())

        self.log_action(f"Created - New catalog '{title}' is now active.")
        self.refresh_all()

    def create_exam_dialog(self):
        dialog = ExamDialog(self)
        if dialog.result:
            self.exam_manager.create_exam(
                title=dialog.result["title"],
                author=dialog.result["author"]
            )
            self.refresh_all()

    def on_catalog_selected(self, event=None):
        """Called when the user selects a catalog from the dropdown."""
        title = self.catalog_var.get()
        if not title:
            return
        self.catalog_manager.set_active_by_title(title)

        self.refresh_all()
        self.log_action(f"Catalog Switched - Active catalog: {title}")

    def on_exam_selected(self, event):
        title = self.target_exam_var.get()
        self.exam_manager.set_active_by_title(title)

    def refresh_exam_list(self):
        titles = self.exam_manager.list_titles()
        self.target_exam_dropdown["values"] = titles
        active_id = self.exam_manager.get_active_id()
        if active_id:
            active_exam = self.exam_manager.get_active()
            self.target_exam_var.set(active_exam.title)

    def get_question_editor_for(parent, question, available_domains):
        """
        Return the appropriate editor dialog instance for a given question.
        This acts as a factory so the main GUI doesn’t need isinstance() checks.
        """
        if question.qtype in ("single_choice", "multiple_choice"):
            return OptionQuestionEditorDialog(parent, question, available_domains)
        if question.qtype in ("instruction", "text"):
            # even though InstructionBlock may subclass Question,
            # it doesn’t need options
            return DefaultQuestionEditorDialog(parent, question, available_domains)
        else:
            # Fallback for any unknown or simple Question subclass
            return DefaultQuestionEditorDialog(parent, question, available_domains)

    def add_question(self):

        active_catalog = self.require_active_catalog()

        if not active_catalog:
            return
        qtype = self.current_qtype_var.get()

        media = [NullMediaItem()]

        if qtype == "single_choice":
            q = SingleChoiceQuestion(text="Neue Frage",
                                     options=[], media=media)
        elif qtype == "multiple_choice":
            q = MultipleChoiceQuestion(text="Neue Mehrfachfrage",
                                       options=[], media=media)
        elif qtype == "instruction":
            q = InstructionBlock(text="Neue Instruktion", media=media)
        else:
            messagebox.showerror("Error", f"Unknown question type: {qtype}")
            return

        dlg = OptionQuestionEditorDialog(self, q, self.domain_set)
        self.wait_window(dlg)
        current_question = dlg.result

        if current_question:
            current_question.flexo_id = FlexOID.generate(
                current_question.domain,
                current_question.entity_type,
                current_question.state,
                current_question.text_seed
            )
            active_catalog.add_questions([current_question])
            self.refresh_all()
            self.log_action(f"Updated - Question {q.id} updated.")
            self.log_action(f"New Question ID - Assigned ID: {current_question.id}")

    def add_selected_question_to_exam(self):
        exam = self.exam_manager.get_active()
        if not exam:
            messagebox.showwarning("No Exam", "Please select or create an exam first.")
            return

        question = self.require_selected_question()
        exam.add_question(question)

    def delete_selected_question(self):
        q = self.require_selected_question()
        if not q:
            return

        if not messagebox.askyesno("Delete", f"Delete question {q.id}?"):
            return

        try:
            if self.active_catalog.remove(q.id):
                self.refresh_all()
                self.log_action(f"Deleted - Question {q.id} removed.")
        except ValueError as e:
            messagebox.showerror("Forbidden", str(e))

    def on_select_question(self, event):
        # Ignore selection events if focus isn’t on the Treeview
        if self.focus_get() is not self.question_tree:
            return
        q = self.require_selected_question()
        if not q:
            return
        catalog = self.catalog_manager.get_active()
        catalog_title = catalog.title if catalog else None
        self.detail_panel.refresh_details(question=q, catalog_title=catalog_title,
                                      clipboard_action=self.clipboard_action,
                                      clipboard_count=len(self.clipboard)
                                      )

    def edit_selected_question(self):
        q = self.require_selected_question()
        dlg = self.get_question_editor_for(q, self.domain_set)
        self.wait_window(dlg)
        current_question = dlg.result
        if current_question:
            self.active_catalog.add_questions([current_question])
            self.refresh_all()
            self.log_action(f"Updated - Question {q.id} updated.")

    def import_exam_as_temp_catalog(self, exam_path: str):
        exam = Exam.from_json_file(exam_path)  # or however you deserialize

        ids = [q.flexo_id for q in exam.questions]
        duplicates = [i for i in set(ids) if ids.count(i) > 1]
        print("Duplicate IDs:", duplicates)

        # Create a temporary catalog
        temp_id = "TEMP_" + datetime.utcnow().strftime("%H%M%S")
        # FIXME: Check flexo_id creation
        temp_catalog = QuestionCatalog(
            title=f"Imported from {Path(exam_path).name}",
            author="import",
            questions=exam.questions
        )
        self.catalog_manager.add_catalog(temp_catalog)
        self.catalog_manager.set_active(temp_id)

        self.refresh_all()
        messagebox.showinfo(
            "Import Complete",
            f"Imported {len(exam.questions)} questions into '{temp_id}'.",
        )
        self.title(f"flex-o-grader – [{self.catalog_manager.get_active_title()}]")
        return exam

    def layout_exam(self):
        exam = self.exam_manager.get_active()
        if not exam:
            messagebox.showinfo("No Exam", "Please select or create an exam first.")
            return
        dlg = ExamLayoutDialog(self, exam)
        self.wait_window(dlg)

    def load_exam(self):
        pass

    def import_exam(self):
        path = filedialog.askopenfilename(filetypes=[("Exam JSON", "*.json")])
        if not path:
            return
        try:
            self.import_exam_as_temp_catalog(path)
        except Exception as e:
            messagebox.showerror("Error", f"Could not load exam:\n{e}")
            return

    def require_selected_question(self):
        """Return the selected Question, or show a message if none is selected or invalid."""
        active = self.require_active_catalog()
        if not active:
            return None

        sel = self.question_tree.selection()
        if not sel:
            messagebox.showinfo("No Selection", "Please select a question first.")
            return None

        qid = sel[0]
        if qid == "__none__":
            return None

        q = active.find(qid)
        if not q:
            messagebox.showerror("Not Found",
                                 f"Question '{qid}' not found in catalog '{active.flexo_id}'.")
            return None
        return q

    def save_question_catalog(
        self, questions, catalog_id, title, output_path, author="unknown"
    ):
        # FIXME: Put this into QuestionCatalog, not Exam
        catalog = self.exam.build_question_catalog(questions, catalog_id, title, author)
        with open(output_path, "w", encoding="utf-8") as f:
            json.dump(catalog, f, indent=2, ensure_ascii=False)
        print(f"Question catalog saved to {output_path}")

    def refresh_question_tree(self):
        """Refresh the non-expandable question list (Treeview)."""
        self.question_tree.delete(*self.question_tree.get_children())

        active = self.require_active_catalog()
        if not active:
            self.question_tree.insert("", "end", iid="__none__",
                                      values=("— No catalog loaded —", ""))
            return

        for q in sorted(active.questions, key=lambda q: q.flexo_id):

            # Clean up the question text (short preview)
            text = q.text.strip().replace("\n", " ")

            # Try to display a readable status name
            try:
                state_name = q.state.name.capitalize().replace("_", " ")
            except AttributeError:
                # fallback if q.state is plain string or missing
                state_name = str(q.state)

            self.question_tree.insert(
                "",
                "end",
                iid=q.flexo_id,                    # keep FlexOID internally for selection
                values=(text, state_name),   # human-readable display
            )

    def _transfer_questions(self, action="copy"):
        sel = self.question_tree.curselection()
        if not sel:
            messagebox.showinfo(action.title(), f"Select a question to {action} first.")
            return

        source_catalog = self.require_active_catalog()
        if not source_catalog:
            return

        catalogs = self.catalog_manager.catalogs()
        if len(catalogs) < 2:
            messagebox.showinfo(action.title(), "You need at least two catalogs.")
            return

        current_id = self.catalog_manager.get_active_id()
        target_choices = [cid for cid in catalogs if cid != current_id]
        if not target_choices:
            messagebox.showinfo(action.title(), "No other catalog available.")
            return

        target_id = simpledialog.askstring(
            f"{action.title()} Question",
            f"Available catalogs:\n{', '.join(target_choices)}\nEnter target catalog ID:",
            parent=self,
        )
        if not target_id:
            return
        if target_id not in self.catalog_manager.catalogs:
            messagebox.showerror(
                action.title(), f"No catalog with ID '{target_id}' found."
            )
            return

        target_catalog = self.catalog_manager.catalogs[target_id]
        selected_indices = list(sel)
        transferred = []

        for idx in reversed(selected_indices):
            q = source_catalog.questions[idx]
            if action == "move":
                q = source_catalog.questions.pop(idx)
            else:
                # shallow copy (no shared object)
                q = copy.deepcopy(q)
            transferred.append(q)

        target_catalog.add_questions(transferred)
        self.refresh_all()

        msg = f"{action.title()}d {len(transferred)} question(s) to catalog '{target_id}'."
        messagebox.showinfo(f"{action.title()} Complete", msg)

    def copy_selected_question(self):
        active = self.require_active_catalog()
        if not active:
            return

        q = self.require_selected_question()
        if not q:
            return

        self.clipboard = copy.deepcopy([self.require_selected_question()])
        self.clipboard_action = "copy"
        self.clipboard_source_catalog = active.flexo_id
        self.log_action(f"Copied {len(self.clipboard)} question copied.")

    # NOTE: I probably remove the function and will display the last copied question in the status bar
    # in case it needs to be deleted in the source catalog
        
    def cut_selected_question(self):
        """Cut (move) the selected question into the clipboard."""
        active = self.require_active_catalog()
        if not active:
            return

        q = self.require_selected_question()
        if not q:
            return

        # Store a *reference* (not a copy) to allow removal from the source
        self.clipboard = [q]
        self.clipboard_action = "cut"
        self.clipboard_source_catalog = str(active.flexo_id)

        self.log_action(f"Cut 1 question from catalog '{active.title}'.")

    def paste_selected_question(self):
        if not self.clipboard:
            messagebox.showinfo("Paste", "Clipboard is empty.")
            return

        target = self.require_active_catalog()
        if not target:
            return

        # If same catalog and copy, warn the user
        if target.flexo_id == self.clipboard_source_catalog and self.clipboard_action == "copy":
            messagebox.showwarning("Paste", "You are pasting into the same catalog.")

        transferred = []

        if self.clipboard_action == "copy":
            # Warn if user tries to copy into the same catalog
            if target.flexo_id == self.clipboard_source_catalog:
                messagebox.showwarning("Paste", "You are pasting into the same catalog.")
                return

            # Deep copy and assign new FlexO-ID
            for q in self.clipboard:
                new_q = copy.deepcopy(q)
                new_q.flexo_id = target.next_question_id(new_q.domain)
                target.add_questions([new_q])
                transferred.append(new_q)
        else:
            messagebox.showwarning("Paste", f"Unknown clipboard action: {self.clipboard_action}")
            return

        # Clear clipboard
        self.clipboard.clear()
        self.clipboard_action = None
        self.clipboard_source_catalog = None

        self.refresh_all()
        self.log_action(f"Pasted {len(transferred)} question(s) to '{target.title}'.")

    def add_domain(self):
        active = self.require_active_catalog()

        if not active:
            return

        entry = simpledialog.askstring("Add Domain",
                                       "Enter domain (e.g. ELEK_Elektrotechnik):",
                                       parent=self)
        if not entry:
            return

        try:
            active.add_domain(entry)
            self.domain_set.update([entry])
            self.log_action(f"Added - Domain '{entry}' added to catalog '{active.catalog_id}'.")
        except ValueError as e:
            messagebox.showerror("Invalid Input", str(e))
            return

    def on_quit(self):
        self.session.save(self.geometry())
        self.destroy()

if __name__ == "__main__":
    app = ExamBuilderApp()
    app.mainloop()
