#!/usr/bin/env python3

import tkinter as tk
from tkinter import ttk


class DetailPanel(ttk.Frame):
    """
    Right-hand info panel for Flex-O-Grader main window.
    Displays question text, metadata, answers, and context (catalog + clipboard).
    """

    def __init__(self, parent):
        super().__init__(parent, padding=5)
        self.grid(row=0, column=1, sticky="nsew")
        self.columnconfigure(0, weight=1)
        self.rowconfigure(3, weight=1)  # answers section expands

        # ─────────────────────────── Question text ───────────────────────────
        ttk.Label(self, text="Question Text").grid(row=0, column=0, sticky="w")
        self.txt_question = tk.Text(self, wrap="word", height=3)
        self.txt_question.grid(row=1, column=0, sticky="ew", pady=(0, 4))

        # ─────────────────────────── Metadata ────────────────────────────────
        meta = ttk.LabelFrame(self, text="Metadata", padding=(6, 4))
        meta.grid(row=2, column=0, sticky="ew", pady=(0, 4))
        meta.columnconfigure(1, weight=1)

        ttk.Label(meta, text="Type:").grid(row=0, column=0, sticky="w")
        self.lbl_type = ttk.Label(meta, text="—")
        self.lbl_type.grid(row=0, column=1, sticky="w")

        ttk.Label(meta, text="State:").grid(row=1, column=0, sticky="w")
        self.lbl_state = ttk.Label(meta, text="—")
        self.lbl_state.grid(row=1, column=1, sticky="w")

        ttk.Label(meta, text="FlexOID:").grid(row=2, column=0, sticky="w")
        self.lbl_id = ttk.Label(meta, text="—")
        self.lbl_id.grid(row=2, column=1, sticky="w")

        ttk.Label(meta, text="Media:").grid(row=3, column=0, sticky="w")
        self.lbl_media = ttk.Label(meta, text="—")
        self.lbl_media.grid(row=3, column=1, sticky="w")

        # ─────────────────────────── Answers table ───────────────────────────
        ttk.Label(self, text="Answers").grid(row=3, column=0, sticky="w")
        self.tree_answers = ttk.Treeview(
            self, columns=("text", "points"), show="headings", height=5
        )
        self.tree_answers.heading("text", text="Answer Text")
        self.tree_answers.heading("points", text="Points")
        self.tree_answers.column("text", width=320, anchor="w")
        self.tree_answers.column("points", width=60, anchor="center")
        self.tree_answers.grid(row=4, column=0, sticky="nsew", pady=(0, 4))

        # ─────────────────────────── Context ────────────────────────────────
        ctx = ttk.LabelFrame(self, text="Context", padding=(6, 4))
        ctx.grid(row=5, column=0, sticky="ew", pady=(4, 0))
        ctx.columnconfigure(1, weight=1)

        ttk.Label(ctx, text="Catalog:").grid(row=0, column=0, sticky="w")
        self.lbl_catalog = ttk.Label(ctx, text="—")
        self.lbl_catalog.grid(row=0, column=1, sticky="w")

        ttk.Label(ctx, text="Clipboard:").grid(row=1, column=0, sticky="w")
        self.lbl_clipboard = ttk.Label(ctx, text="Empty")
        self.lbl_clipboard.grid(row=1, column=1, sticky="w")

    # ───────────────────────────────────────────────────────────────────────
    def refresh_details(self, question=None, catalog_title=None,
                    clipboard_action=None, clipboard_count=0):
        """
        Refresh all visible info according to the selected question and state.
        Called by main GUI after question selection or clipboard change.
        """
        # Clear / update question text
        self.txt_question.delete("1.0", tk.END)
        if question:
            self.txt_question.insert("1.0", question.text)
            self.lbl_type.config(text=question.__class__.__name__)
            try:
                self.lbl_state.config(text=question.state.name)
            except AttributeError:
                self.lbl_state.config(text=str(question.state))
            self.lbl_id.config(text=str(question.flexo_id))
            self.lbl_media.config(
                text=", ".join([m.src for m in question.media if m]) or "—"
            )
        else:
            self.lbl_type.config(text="—")
            self.lbl_state.config(text="—")
            self.lbl_id.config(text="—")
            self.lbl_media.config(text="—")

        # Update answers
        for item in self.tree_answers.get_children():
            self.tree_answers.delete(item)
        if question:
            for text, points in question.answer_options_for_display():
                self.tree_answers.insert("", "end", values=(text, points))

        # Context: catalog & clipboard
        self.lbl_catalog.config(text=catalog_title or "—")
        if clipboard_count == 0:
            self.lbl_clipboard.config(text="Empty")
        else:
            self.lbl_clipboard.config(
                text=f"{clipboard_action or '?'} ({clipboard_count})"
            )
