from typing import List, Dict, Any

from .exam_elements import (
    SingleChoiceQuestion,
    MultipleChoiceQuestion,
    TextQuestion,
    CandidateIDQuestion,
    InstructionBlock,
    Validation,
    AnswerOption,
)
from .media_items import media_factory, NullMediaItem
from flexoentity import EntityType, EntityState, FlexOID, Domain


# ──────────────────────────────────────────────────────────────────────────────
# Helpers
# ──────────────────────────────────────────────────────────────────────────────

def _ensure_flexoid_and_state(q: dict, default_domain: str) -> tuple[FlexOID, EntityState, dict]:
    """
    I return (flexoid, state, parsed) for the question dict.
    - If 'flexo_id' present: parse it.
    - Else: generate a fresh one with DRAFT state using domain/text.
    """
    if q.get("flexo_id"):
        oid = FlexOID(q["flexo_id"])
        p = oid.parsed()
        if not p["state"]:
            state = EntityState.DRAFT
        else:
            state = EntityState(p["state"])
        return oid, state, p

    domain = q.get("domain") or default_domain
    text = q.get("text", "")
    estate = EntityState.DRAFT
    oid = FlexOID.generate(domain, "I", estate.value, text, version=1)
    return oid, EntityState.DRAFT, oid.parsed()


def options_factory(opt_list: List[Dict[str, Any]]) -> List[AnswerOption]:
    return [AnswerOption(o["id"], o["text"], o.get("points", 0.0)) for o in opt_list]

def _media_list(q: dict):
    media = [media_factory(m) for m in q.get("media", [])]
    return media or [NullMediaItem.default()]


# ──────────────────────────────────────────────────────────────────────────────
# Question builders
# ──────────────────────────────────────────────────────────────────────────────

class QuestionTypes:
    @staticmethod
    def single_choice(q: dict):
        oid, state, p = _ensure_flexoid_and_state(q, default_domain=Domain.default().domain_code)
        q = SingleChoiceQuestion(
            text=q.get("text", ""),
            topic=q.get("topic", ""),
            options=options_factory(q.get("options", [])),
            media=_media_list(q),
        )
        q.flexo_id = oid
        return q

    @staticmethod
    def multiple_choice(q: dict):
        oid, state, p = _ensure_flexoid_and_state(q, default_domain=Domain.default().domain_code)
        q = MultipleChoiceQuestion(
            text=q.get("text", ""),
            topic=q.get("topic", ""),
            options=options_factory(q.get("options", [])),
            media=_media_list(q),
        )
        q.flexo_id = oid
        return q

    @staticmethod
    def text(q: dict):
        oid, state, p = _ensure_flexoid_and_state(q, default_domain=Domain.default().domain_code)
        val = Validation(**q["validation"]) if "validation" in q else None
        q = TextQuestion(
                text=q.get("text", ""),
                topic=q.get("topic", ""),
                validation=val,
                media=_media_list(q),
        )
        q.flexo_id = oid
        return q

    @staticmethod
    def candidate_id(q: dict):
        oid, state, p = _ensure_flexoid_and_state(q, default_domain="IDENT")
        q = CandidateIDQuestion(
                text=q.get("text", ""),
                topic=q.get("topic", ""),
                fields=q.get("fields", []),
                media=_media_list(q),
            )
        q.flexo_id = oid
        return q

    @staticmethod
    def instruction(q: dict):
        oid, state, p = _ensure_flexoid_and_state(q, default_domain="INFO")
        q = InstructionBlock(
                text=q.get("text", ""),
                topic=q.get("topic", ""),
                media=_media_list(q),
            )
        q.flexo_id = oid
        return q

# ──────────────────────────────────────────────────────────────────────────────
# Public factory
# ──────────────────────────────────────────────────────────────────────────────

def question_factory(q: dict):
    qtype = q["qtype"]
    try:
        return getattr(QuestionTypes, qtype)(q)
    except AttributeError as e:
        raise ValueError(f"Unknown question type: {qtype}") from e
