import base64
from dataclasses import dataclass
from typing import Optional
from uuid import UUID
from flexoentity import FlexoEntity, FlexOID, EntityType


@dataclass
class CertificateReference:
    """
    A cross-platform reference to a signing certificate.

    This does NOT contain private keys or secret material.
    It only contains the information required for OS-provided APIs
    (OpenSSL, certutil, security) to *locate* or *verify* the certificate.

    Fields:
      platform:
         Optional override for backend creation.
         Values: "LINUX", "WINDOWS", "MACOS" (case-insensitive).
         If None, platform.system() determines the backend.

      identifier:
         Linux:   path to certificate PEM file
         Windows: certificate thumbprint (SHA1, with or without colons)
         macOS:   certificate Common Name (CN) in Keychain

      private_key_path:
         Linux only:
            path to private key PEM file.
         Windows/macOS:
            MUST be None.

      public_cert_path:
         The path to the public certificate for verification.
         Linux:
            optional (defaults to `identifier`)
         Windows:
            optional; used if you want to validate against a specific cert
         macOS:
            REQUIRED (used for OpenSSL verification)
    """

    platform: Optional[str] = None
    identifier: str = ""
    private_key_path: Optional[str] = None
    public_cert_path: Optional[str] = None

    def to_dict(self):
        return {
            "platform": self.platform,
            "identifier": self.identifier,
            "private_key_path": self.private_key_path,
            "public_cert_path": self.public_cert_path
        }


@dataclass
class FlexoSignature(FlexoEntity):
    """
    I represent a digital or procedural signature for another entity.

    This is a stub version: I only carry logical metadata, not cryptographic proof.
    Later, platform-specific implementations (e.g. Windows certificate signing)
    can fill the 'signature_data' field with real data.

    Lifecycle:
      - Created in DRAFT → becomes APPROVED once verified
      - Optionally moves to PUBLISHED if distributed externally
    """
    ENTITY_TYPE = EntityType.ATTESTATION

    signed_entity: Optional[FlexOID] = None
    signer_id: Optional[UUID] = None

    # PKCS#7 DER, base64-encoded
    signature_data: str = ""
    signature_type: str = "PKCS7-DER"

    certificate_reference: CertificateReference | None = None
    certificate_thumbprint: str = ""

    comment: Optional[str] = None

    def _serialize_content(self):
        return {
            "signed_entity": str(self.signed_entity),
            "signer_id": str(self.signer_id),
            "signature_data": self.signature_data,
            "signature_type": self.signature_type,
            # "certificate_reference": self.certificate_reference.to_dict(),
            "certificate_thumbprint": self.certificate_thumbprint,
            "comment": self.comment
        }

    @property
    def text_seed(self):
        return f"{self.signed_entity}:{self.signer_id}:{self.certificate_thumbprint}"

    @classmethod
    def default(cls):
        """Required by FlexoEntity. Returns an empty draft signature."""
        return cls()

    @classmethod
    def create_signed(cls, data: bytes, entity: FlexOID, signer_id: UUID, backend):
        sig = backend.sign(data)
        return cls.with_domain_id(
            domain_id=entity.domain_id,
            signed_entity=entity,
            signer_id=signer_id,
            signature_data=base64.b64encode(sig).decode(),
            certificate_reference=backend.cert_ref,
            certificate_thumbprint=backend.certificate_thumbprint
        )

    def verify(self, data: bytes, backend) -> bool:
        raw = base64.b64decode(self.signature_data)
        return backend.verify(data, raw)

