from .persistance_backend import PersistanceBackend
from .flexo_entity import FlexoEntity
from .in_memory_backend import InMemoryBackend

class CompositeBackend(PersistanceBackend):
    """
    A backend that wraps multiple real backends.

    - Reads always come from the first backend (e.g., InMemoryBackend).
    - Writes propagate to all backends.
    """

    def __init__(self, authoritative_backend, sync_backends):
        if not issubclass(authoritative_backend.entity_class, FlexoEntity):
            raise TypeError("entity_class must be a subclass of FlexOEntity")

        self._primary = authoritative_backend

        self.read_backend = InMemoryBackend(self.primary.entity_class)
        # Default: create an in-memory backend as the primary backend

        if sync_backends is None:
            self.sync_backends = []
        else:
            self.sync_backends = sync_backends

        # Validate all backends
        for b in self.sync_backends:
            if b.entity_class != self.primary.entity_class:
                raise TypeError(
                    f"Backend {b} does not match entity_class={self.entity_class.__name__}"
                )

    @property
    def primary(self):
        """The backend used for all read operations."""
        return self._primary

    @primary.setter
    def primary(self, new_primary):
        self._primary = new_primary    

    def add_backend(self, backend, clear=False):
        """
        Append an additional backend.
        If clear=True, backend is wiped before syncing.
        """
        if backend.entity_class != self.primary.entity_class:
            raise TypeError("Backend entity_class mismatch")

        if clear:
            backend.delete_all()

        # Sync current data into backend
        for entity in self.primary.load_all():
            backend.save(entity)

        self.sync_backends.append(backend)

    def remove_backend(self, backend):
        """
        Remove a backend. Primary backend cannot be removed.
        """
        if backend is self.primary:
            raise ValueError("Cannot remove the primary backend")
        self.sync_backends.remove(backend)
    # ---------------------------------------------------------
    # Write operations propagate to *all* backends
    # ---------------------------------------------------------

    def save(self, entity):
        self.primary.save(entity)
        for b in self.sync_backends:
            b.save(entity)

    def update(self, entity):
        for b in self.sync_backends:
            b.update(entity)

    def delete(self, flexo_id: str):
        for b in self.sync_backends:
            b.delete(flexo_id)

    # ---------------------------------------------------------
    # Read operations use only the *primary* backend
    # ---------------------------------------------------------

    def load(self, flexo_id: str):
        return self.primary.load(flexo_id)

    def load_all(self):
        return self.primary.load_all()

    # ---------------------------------------------------------
    # Optional: flush from primary backend to all others
    # ---------------------------------------------------------

    def sync_all(self):
        """
        Push all data from the primary backend to the other backends.
        Useful if secondary backends were empty initially.
        """
        for entity in primary.load_all():
            for b in self.sync_backends:
                b.save(entity)

    def clear(self):
        self.primary.clear()
        for b in self.sync_backends:
            if hasattr(b, "clear"):
                b.clear()

    # ---------------------------------------------------------

    def __repr__(self):
        names = ", ".join(b.__class__.__name__ for b in self.sync_backends)
        return f"<CompositeBackend [{names}]>"
