from __future__ import annotations
import json

from typing import Iterable

from .flexo_collection import FlexoCollection
from .domain import Domain
from .flexo_entity import EntityState
from .entity_manager import EntityManager


class DomainNotFoundError(KeyError):
    pass


class InvalidDomainError(ValueError):
    pass


class DuplicateDomainError(ValueError):
    pass


class DomainManager(EntityManager):
    """
    Manager for Domain entities using the new backend architecture.

    - Primary storage: backend.load_all() (InMemory, SQLite, JSON, ...)
    - Secondary index: domain_id → Domain
    - Domain-specific business rules included.
    """

    ENTITY_CLASS = Domain

    def __init__(self, local_backend, staging_backend, permanent_backend, registry=None):
        super().__init__(local_backend=local_backend,
                         staging_backend=staging_backend,
                         permanent_backend=permanent_backend)
        self._registry = registry
        self._index_by_domain_id = {}

        self._rebuild_index()

    def refresh(self):
        self._rebuild_index

    # ------------------------------------------------------------
    # Index rebuild
    # ------------------------------------------------------------

    def _rebuild_index(self):
        """Rebuild domain_id lookup table from backend contents."""
        self._index_by_domain_id.clear()
        for backend in [self.local_backend, self.staging_backend, self.permanent_backend]:
            for dom in backend.load_all():
                self._index_by_domain_id[dom.domain_id] = dom

    # ------------------------------------------------------------
    # Overrides for CRUD
    # ------------------------------------------------------------

    def add(self, domain: Domain):
        # Domain-specific uniqueness rule
        if domain.domain_id in self._index_by_domain_id:
            raise DuplicateDomainError(f"Domain '{domain.domain_id}' already exists.")

        super().add(domain)
        self._index_by_domain_id[domain.domain_id] = domain

    def update(self, domain: Domain):
        super().update(domain)
        self._index_by_domain_id[domain.domain_id] = domain

    def delete(self, flexo_id: str):
        dom = self.local_backend.load(flexo_id)
        if dom is None:
            return

        domain_id = dom.domain_id

        # Check domain deletion rules
        if not self.can_delete(domain_id):
            raise ValueError(f"Domain '{domain_id}' is still referenced.")

        super().delete(flexo_id)

        # Remove from index
        self._index_by_domain_id.pop(domain_id, None)

    # ------------------------------------------------------------
    # Domain-specific lookup
    # ------------------------------------------------------------

    def get_by_domain_id(self, domain_id: str) -> Domain:
        dom = self._index_by_domain_id.get(domain_id)
        if not dom:
            raise KeyError(f"Domain '{domain_id}' not found.")
        return dom

    def find_by_domain_id(self, domain_id: str):
        return self._index_by_domain_id.get(domain_id)

    def all_domain_ids(self):
        return list(self._index_by_domain_id.keys())

    def all(self):
        return list(self._index_by_domain_id.values())
    # ------------------------------------------------------------
    # Domain deletion rules
    # ------------------------------------------------------------

    def can_delete(self, domain_id: str) -> bool:
        if not self._registry:
            return True
        return not any(self._registry.entities_by_domain(domain_id))

    # ------------------------------------------------------------
    # Domain lifecycle helpers
    # ------------------------------------------------------------

    def ensure_approved(self, domain_id: str) -> None:
        dom = self.get_by_domain_id(domain_id)

        if dom.state != EntityState.APPROVED:
            raise ValueError(
                f"Domain '{domain_id}' must be APPROVED before "
                f"entities in this domain can be promoted. "
                f"(Current state: {dom.state})"
            )

    def clear(self):
        self.local_backend.clear()

    def __repr__(self):
        return f"<DomainManager domains={self.all_domain_ids()}>"
