from uuid import UUID
from dataclasses import dataclass
from flexoentity import FlexOID, FlexoEntity, EntityType

@dataclass
class Domain(FlexoEntity):
    """
    I am a helper class to provide more information than just a
    domain abbreviation in FlexOID, doing mapping and management
    """

    ENTITY_TYPE = EntityType.DOMAIN

    fullname: str = ""
    description: str = ""
    classification: str = "UNCLASSIFIED"

    @classmethod
    def default(cls):
        """Return the default domain object."""
        return cls.with_domain_id(domain_id="GEN_GENERIC",
                                  fullname="Generic Domain", classification="UNCLASSIFIED")

    @property
    def text_seed(self) -> str:
        return self.domain_id

    def to_dict(self):
        base = super().to_dict()
        base.update({
            "flexo_id": self.flexo_id,
            "domain_id": self.domain_id,
            "fullname": self.fullname,
            "description": self.description,
            "classification": self.classification,
        })
        return base

    @classmethod
    def from_dict(cls, data):
        # Must have flexo_id
        if "flexo_id" not in data:
            raise ValueError("Domain serialization missing 'flexo_id'.")

        flexo_id = FlexOID(data["flexo_id"])

        obj = cls(
            fullname=data.get("fullname", ""),
            description=data.get("description", ""),
            classification=data.get("classification", "UNCLASSIFIED"),
            flexo_id=flexo_id,
            _in_factory=True
        )

        # Restore metadata
        obj.origin = data.get("origin")
        obj.fingerprint = data.get("fingerprint", "")
        obj.originator_id = (
            UUID(data["originator_id"]) if data.get("originator_id") else None
        )
        obj.owner_id = (
            UUID(data["owner_id"]) if data.get("owner_id") else None
        )

        return obj
