import platform
import pytest
from uuid import uuid4
from flexoentity import FlexOID, EntityState, EntityType, FlexoSignature, get_signing_backend

def test_sign_and_verify_linux(backend):
    data = b"Hello Flex-O signing!"

    signature = backend.sign(data)
    assert isinstance(signature, bytes)
    assert len(signature) > 20      # sanity check

    assert backend.verify(data, signature) is True

def test_sign_and_verify_macos(backend):
    data = b"Hello Flex-O signing!"

    signature = backend.sign(data)
    assert isinstance(signature, bytes)
    assert len(signature) > 20      # sanity check

    assert backend.verify(data, signature) is True

def test_verify_fails_with_wrong_data(backend):
    data = b"Original Data"
    wrong_data = b"Tampered Data"

    signature = backend.sign(data)

    assert backend.verify(data, signature) == True
    assert backend.verify(wrong_data, signature) == False

def test_verify_fails_with_invalid_signature(backend):
    data = b"Hello world"
    invalid_sig = b"\x00\x01\x02garbagepkcs7data"

    assert backend.verify(data, invalid_sig) is False

def test_signature_entity_create_and_verify(backend):
    entity_id = FlexOID.safe_generate(
        domain_id="TEST",
        entity_type=EntityType.ATTESTATION.value,
        state=EntityState.DRAFT.value,
        text="abc",
        version=1,
    )
    signer = uuid4()
    data = b"Hello Entity Signing"

    sig = FlexoSignature.create_signed(
        data=data,
        entity=entity_id,
        signer_id=signer,
        backend=backend,
    )

    assert isinstance(sig.signature_data, str)
    assert sig.signature_type == "PKCS7-DER"
    assert sig.signed_entity == entity_id
    assert sig.signer_id == signer
    assert sig.certificate_thumbprint != ""

    assert sig.verify(data, backend) is True
