import subprocess
import platform
from pathlib import Path
import tempfile
import uuid
from abc import ABC, abstractmethod

class SigningBackend(ABC):
    def __init__(self, cert_ref):
        """
        cert_ref is an object that is expected to provide (depending on platform):

        Linux:
          - identifier: path to certificate (PEM)
          - private_key_path: path to private key (PEM)
          - public_cert_path: optional, defaults to identifier if missing

        Windows:
          - identifier: certificate thumbprint (SHA1; with or without colons)

        macOS:
          - identifier: certificate Common Name (CN)
          - public_cert_path: path to public certificate (PEM)
        """
        self.cert_ref = cert_ref

    @property
    @abstractmethod
    def certificate_thumbprint(self) -> str:
        """Return canonical thumbprint for Signature entity storage."""
        raise NotImplementedError

    @abstractmethod
    def sign(self, data: bytes) -> bytes:
        """Return PKCS#7 DER signature bytes."""
        raise NotImplementedError

    @abstractmethod
    def verify(self, data: bytes, signature: bytes) -> bool:
        """Return True/False after verifying PKCS#7 signature."""
        raise NotImplementedError

    def _tmp(self, suffix: str) -> Path:
        """Return a unique temporary file path for this process."""
        return Path(tempfile.NamedTemporaryFile(delete=False).name).with_suffix(suffix)

    
class LinuxOpenSSLBackend(SigningBackend):

    @property
    def certificate_thumbprint(self) -> str:
        out = subprocess.check_output([
            "openssl", "x509", "-in", self.cert_ref.identifier,
            "-noout", "-fingerprint", "-sha1"
        ]).decode()
        return out.strip().split("=")[1].replace(":", "")

    def sign(self, data: bytes) -> bytes:
        datafile = self._tmp(".bin")
        sigfile = self._tmp(".p7s")
        datafile.write_bytes(data)

        subprocess.check_call([
            "openssl", "cms", "-sign",
            "-binary",
            "-in", str(datafile),
            "-signer", self.cert_ref.identifier,
            "-inkey", self.cert_ref.private_key_path,     # operator-provided
            "-outform", "DER",
            "-out", str(sigfile)
        ])

        out = sigfile.read_bytes()
        datafile.unlink()
        sigfile.unlink()
        return out

    def verify(self, data: bytes, signature: bytes) -> bool:
        datafile = self._tmp(".bin")
        sigfile  = self._tmp(".p7s")
        datafile.write_bytes(data)
        sigfile.write_bytes(signature)

        try:
            subprocess.check_call([
                "openssl", "cms", "-verify",
                "-in", str(sigfile),
                "-inform", "DER",
                "-content", str(datafile),
                "-CAfile", self.cert_ref.public_cert_path,
                "-purpose", "any",
                "-out", "/dev/null",
            ])
            return True
        except subprocess.CalledProcessError:
            return False
        finally:
            datafile.unlink()
            sigfile.unlink()

class WindowsCertutilBackend(SigningBackend):

    @property
    def certificate_thumbprint(self) -> str:
        return self.cert_ref.identifier.upper().replace(":", "")

    def sign(self, data: bytes) -> bytes:
        datafile = self._tmp(".bin")
        sigfile  = self._tmp(".p7s")
        datafile.write_bytes(data)

        cmd = [
            "certutil",
            "-user",
            "-silent",
            "-p", "",
            "-sign",
            str(datafile),
            str(sigfile),
            self.certificate_thumbprint
        ]

        subprocess.check_call(cmd)
        out = sigfile.read_bytes()
        datafile.unlink()
        sigfile.unlink()
        return out

    def verify(self, data: bytes, signature: bytes) -> bool:
        datafile = self._tmp(".bin")
        sigfile = self._tmp(".p7s")
        datafile.write_bytes(data)
        sigfile.write_bytes(signature)

        try:
            subprocess.check_call([
                "certutil", "-verify", str(sigfile), str(datafile)
            ])
            return True
        except subprocess.CalledProcessError:
            return False
        finally:
            datafile.unlink()
            sigfile.unlink()

class MacOSSecurityCMSBackend(SigningBackend):

    @property
    def certificate_thumbprint(self) -> str:
        # Derive from provided public cert
        out = subprocess.check_output([
            "openssl", "x509", "-in", self.cert_ref.public_cert_path,
            "-noout", "-fingerprint", "-sha1"
        ]).decode()
        return out.strip().split("=")[1].replace(":", "")

    def sign(self, data: bytes) -> bytes:
        datafile = self._tmp(".bin")
        sigfile = self._tmp(".p7s")
        datafile.write_bytes(data)

        subprocess.check_call([
            "security", "cms", "-S",
            "-N", self.cert_ref.identifier,   # CN
            "-i", str(datafile),
            "-o", str(sigfile)
        ])

        out = sigfile.read_bytes()
        datafile.unlink()
        sigfile.unlink()
        return out

    def verify(self, data: bytes, signature: bytes) -> bool:
        sigfile = self._tmp(".p7s")
        datafile = self._tmp(".bin")
        sigfile.write_bytes(signature)
        datafile.write_bytes(data)

        # explicit verify (preferred)
        if self.cert_ref.public_cert_path:
            try:
                subprocess.check_call([
                    "openssl", "cms", "-verify",
                    "-in", str(sigfile),
                    "-inform", "DER",
                    "-content", str(datafile),
                    "-CAfile", self.cert_ref.public_cert_path,
                    "-purpose", "any",
                    "-out", "/dev/null",
                ])
                return True
            except subprocess.CalledProcessError:
                return False

        # fallback: structural validation only
        try:
            subprocess.check_call(["security", "cms", "-D", "-i", str(sigfile)])
            return True
        except subprocess.CalledProcessError:
            return False
        finally:
            sigfile.unlink()
            datafile.unlink()

def get_signing_backend(cert_ref):
    system = platform.system()
    if system == "Linux":
        return LinuxOpenSSLBackend(cert_ref)
    if system == "Windows":
        return WindowsCertutilBackend(cert_ref)
    if system == "Darwin":
        return MacOSSecurityCMSBackend(cert_ref)
    raise RuntimeError("Unsupported platform")
