
# Table of Contents

-   [Overview](#org8360697)
-   [Flex-O ID Structure](#org89396ad)
-   [Lifecycle States](#orgd522c8c)
-   [Core Classes](#org381a3f6)
    -   [FlexOID](#org6abdc86)
    -   [`FlexoEntity`](#org15b9543)
-   [Integrity Verification](#org1617277)
-   [Real World Example](#orge08e5d3)
-   [Usage](#orgc8e7dd2)
-   [Serialization Example](#org1b3bbed)
-   [Entity Type and State Codes](#org5dfc109)
-   [Design Notes](#org659c733)
-   [Dependencies](#orga67f3a6)
-   [Integration](#org10ab165)
-   [License](#org48ba119)



<a id="org8360697"></a>

# Overview

\`flexoentity\` provides the **identity and lifecycle backbone** for all Flex-O components  
(Flex-O-Grader, Flex-O-Vault, Flex-O-Drill, …).

It defines how entities such as questions, media, catalogs, and exams are **identified, versioned, signed, and verified** — all without any external database dependencies.

At its heart lie two modules:

-   `id_factory.py` – robust, cryptographically-verifiable **Flex-O ID generator**
-   `flexo_entity.py` – abstract **base class for all versioned entities**

Together, they form a compact yet powerful foundation for audit-ready, reproducible data structures across offline and air-gapped deployments.

-   Design Goals

<table border="2" cellspacing="0" cellpadding="6" rules="groups" frame="hsides">


<colgroup>
<col  class="org-left" />

<col  class="org-left" />
</colgroup>
<thead>
<tr>
<th scope="col" class="org-left">Goal</th>
<th scope="col" class="org-left">Description</th>
</tr>
</thead>
<tbody>
<tr>
<td class="org-left"><b>Determinism</b></td>
<td class="org-left">IDs are derived from canonicalized entity content — identical input always yields identical ID prefix.</td>
</tr>

<tr>
<td class="org-left"><b>Integrity</b></td>
<td class="org-left">BLAKE2s hashing and digital signatures protect against manual tampering.</td>
</tr>

<tr>
<td class="org-left"><b>Traceability</b></td>
<td class="org-left">Version numbers (<code>@001A</code>, <code>@002S</code>, …) track entity lifecycle transitions.</td>
</tr>

<tr>
<td class="org-left"><b>Stability</b></td>
<td class="org-left">Hash prefixes remain constant across state changes; only version and state suffixes evolve.</td>
</tr>

<tr>
<td class="org-left"><b>Auditability</b></td>
<td class="org-left">Every entity can be serialized, verified, and reconstructed without hidden dependencies.</td>
</tr>

<tr>
<td class="org-left"><b>Simplicity</b></td>
<td class="org-left">Pure-Python, zero external libraries, self-contained and easy to embed.</td>
</tr>
</tbody>
</table>


<a id="org89396ad"></a>

# Flex-O ID Structure

Each entity carries a unique **Flex-O ID**, generated by `FlexOID.generate()`.

    AF-I250101-9A4C2D@003S

<table border="2" cellspacing="0" cellpadding="6" rules="groups" frame="hsides">


<colgroup>
<col  class="org-left" />

<col  class="org-left" />

<col  class="org-left" />
</colgroup>
<thead>
<tr>
<th scope="col" class="org-left">Segment</th>
<th scope="col" class="org-left">Example</th>
<th scope="col" class="org-left">Meaning</th>
</tr>
</thead>
<tbody>
<tr>
<td class="org-left"><b>Domain</b></td>
<td class="org-left"><code>AF or PY_LANG</code></td>
<td class="org-left">Uppercase - Logical scope (e.g. &ldquo;Air Force&rdquo;)</td>
</tr>

<tr>
<td class="org-left"><b>Type</b></td>
<td class="org-left"><code>I</code></td>
<td class="org-left">Entity type (e.g. ITEM)</td>
</tr>

<tr>
<td class="org-left"><b>Date</b></td>
<td class="org-left"><code>250101</code></td>
<td class="org-left">UTC creation date (YYMMDD)</td>
</tr>

<tr>
<td class="org-left"><b>Hash</b></td>
<td class="org-left"><code>9A4C2D4F6E53</code></td>
<td class="org-left">12-digit BLAKE2s digest of canonical content</td>
</tr>

<tr>
<td class="org-left"><b>Version</b></td>
<td class="org-left"><code>003</code></td>
<td class="org-left">Sequential version counter</td>
</tr>

<tr>
<td class="org-left"><b>State</b></td>
<td class="org-left"><code>S</code></td>
<td class="org-left">Lifecycle state (D, A, S, P, O)</td>
</tr>
</tbody>
</table>


<a id="orgd522c8c"></a>

# Lifecycle States

<table border="2" cellspacing="0" cellpadding="6" rules="groups" frame="hsides">


<colgroup>
<col  class="org-left" />

<col  class="org-left" />

<col  class="org-left" />
</colgroup>
<thead>
<tr>
<th scope="col" class="org-left">State</th>
<th scope="col" class="org-left">Abbrev.</th>
<th scope="col" class="org-left">Description</th>
</tr>
</thead>
<tbody>
<tr>
<td class="org-left"><b>DRAFT</b></td>
<td class="org-left"><code>D</code></td>
<td class="org-left">Editable, not yet validated</td>
</tr>

<tr>
<td class="org-left"><b>APPROVED</b></td>
<td class="org-left"><code>A</code></td>
<td class="org-left">Reviewed and accepted</td>
</tr>

<tr>
<td class="org-left"><b>APPROVED<sub>AND</sub><sub>SIGNED</sub></b></td>
<td class="org-left"><code>S</code></td>
<td class="org-left">Cryptographically signed</td>
</tr>

<tr>
<td class="org-left"><b>PUBLISHED</b></td>
<td class="org-left"><code>P</code></td>
<td class="org-left">Released to consumers</td>
</tr>

<tr>
<td class="org-left"><b>OBSOLETE</b></td>
<td class="org-left"><code>O</code></td>
<td class="org-left">Archived or replaced</td>
</tr>
</tbody>
</table>

Transitions follow a strict progression:

    DRAFT -> APPROVED -> APPROVED_AND_SIGNED -> PUBLISHED -> OBSOLETE

Only DRAFT entities can be deleted - all others got OBSOLETE mark instead


<a id="org381a3f6"></a>

# Core Classes


<a id="org6abdc86"></a>

## FlexOID

A lightweight immutable class representing the full identity of an entity.

**Highlights**

-   safe<sub>generate</sub>(domain, entity<sub>type</sub>, estate, text, version=1, repo) -> create a new ID
-   next<sub>version</sub>(oid) -> increment version safely
-   clone<sub>new</sub><sub>base</sub>(domain, entity<sub>type</sub>, estate, text) -> start a new lineage
-   Deterministic prefix, state-dependent signature


<a id="org15b9543"></a>

## `FlexoEntity`

Abstract base class for all versioned entities (e.g., Question, Exam, Catalog).

Implements:

-   ID lifecycle management (approve(), sign(), publish(), obsolete())
-   Serialization (to<sub>json</sub>(), from<sub>json</sub>(), to<sub>dict</sub>(), from<sub>dict</sub>())
-   Integrity verification (verify<sub>integrity</sub>(entity))
-   Controlled state transitions with automatic timestamps

Subclasses define a single property:

    @property
    def text_seed(self) -> str:
        """Canonical text or core content for hashing."""


<a id="org1617277"></a>

# Integrity Verification

Each entity can self-verify its integrity:

    entity = Question.with_domain_id(domain_id="AF", text="What is Ohm’s law?", topic="Electronics")
    json_str = entity.to_json()
    reloaded = Question.from_json(json_str)
    
    assert FlexoEntity.verify_integrity(reloaded)

If the file is tampered with (e.g. &ldquo;Ohm’s&rdquo; → &ldquo;Omm’s&rdquo;), verification fails:


<a id="orge08e5d3"></a>

# Real World Example

Below you can see the implementation of a dedicated FlexoEntity class, used for Domains.
We set an ENTITY<sub>TYPE</sub> and define the needed fields in the data class. We define how to create
a default object, the text<sub>seed</sub> (it is easy because the domain id is unique and therefore sufficient)
and the methods for serialization.

    from uuid import UUID
    from dataclasses import dataclass
    from flexoentity import FlexOID, FlexoEntity, EntityType
    
    @dataclass
    class Domain(FlexoEntity):
        """
        I am a helper class to provide more information than just a
        domain abbreviation in FlexOID, doing mapping and management
        """
    
        ENTITY_TYPE = EntityType.DOMAIN
    
        fullname: str = ""
        description: str = ""
        classification: str = "UNCLASSIFIED"
    
        @classmethod
        def default(cls):
            """Return the default domain object."""
            return cls.with_domain_id(domain_id="GEN_GENERIC",
                                      fullname="Generic Domain", classification="UNCLASSIFIED")
    
        @property
        def text_seed(self) -> str:
            return self.domain_id
    
        def to_dict(self):
            base = super().to_dict()
            base.update({
                "flexo_id": self.flexo_id,
                "domain_id": self.domain_id,
                "fullname": self.fullname,
                "description": self.description,
                "classification": self.classification,
            })
            return base
    
        @classmethod
        def from_dict(cls, data):
            # Must have flexo_id
            if "flexo_id" not in data:
                raise ValueError("Domain serialization missing 'flexo_id'.")
    
            flexo_id = FlexOID(data["flexo_id"])
    
            obj = cls(
                fullname=data.get("fullname", ""),
                description=data.get("description", ""),
                classification=data.get("classification", "UNCLASSIFIED"),
                flexo_id=flexo_id,
                _in_factory=True
            )
    
            # Restore metadata
            obj.origin = data.get("origin")
            obj.fingerprint = data.get("fingerprint", "")
            obj.originator_id = (
                UUID(data["originator_id"]) if data.get("originator_id") else None
            )
            obj.owner_id = (
                UUID(data["owner_id"]) if data.get("owner_id") else None
            )
    
            return obj


<a id="orgc8e7dd2"></a>

# Usage

    d = Domain.default()
    print(d.flexo_id)             # GEN_GENERIC-D251124-67C2CAE292CE@001D
    d.approve()
    print(d.flexo_id)             # GEN_GENERIC-D251124-67C2CAE292CE@001A
    d.sign()
    print(d.flexo_id)             # GEN_GENERIC-D251124-67C2CAE292CE@001S


<a id="org1b3bbed"></a>

# Serialization Example

    {
        'flexo_id': FlexOID(GEN_GENERIC-D251124-29CE0F4BE59D@001S),
        'fingerprint': '534BD2EC5C5511F1',
        'origin': FlexOID(GEN_GENERIC-D251124-67C2CAE292CE@001D),
        'originator_id': '00000000-0000-0000-0000-000000000000',
        'owner_id': '00000000-0000-0000-0000-000000000000',
        'domain_id': 'GEN_GENERIC',
        'fullname': 'Generic Domain',
        'description': '',
        'classification': 'UNCLASSIFIED'}

\#+BEGIN<sub>SRC</sub> js

{
        &ldquo;flexo<sub>id</sub>&rdquo;: &ldquo;GEN<sub>GENERIC</sub>-D251124-29CE0F4BE59D@001S&rdquo;,
        &ldquo;fingerprint&rdquo;: &ldquo;534BD2EC5C5511F1&rdquo;,
        &ldquo;origin&rdquo;: &ldquo;GEN<sub>GENERIC</sub>-D251124-67C2CAE292CE@001D&rdquo;,
        &ldquo;originator<sub>id</sub>&rdquo;: &ldquo;00000000-0000-0000-0000-000000000000&rdquo;,
        &ldquo;owner<sub>id</sub>&rdquo;: &ldquo;00000000-0000-0000-0000-000000000000&rdquo;,
        &ldquo;domain<sub>id</sub>&rdquo;: &ldquo;GEN<sub>GENERIC</sub>&rdquo;,
        &ldquo;fullname&rdquo;: &ldquo;Generic Domain&rdquo;,
        &ldquo;description&rdquo;: &ldquo;&rdquo;,
        &ldquo;classification&rdquo;: &ldquo;UNCLASSIFIED&rdquo;
}


<a id="org5dfc109"></a>

# Entity Type and State Codes

<table border="2" cellspacing="0" cellpadding="6" rules="groups" frame="hsides">


<colgroup>
<col  class="org-left" />

<col  class="org-left" />

<col  class="org-left" />
</colgroup>
<thead>
<tr>
<th scope="col" class="org-left">EntityType</th>
<th scope="col" class="org-left">Code</th>
<th scope="col" class="org-left">Typical Use</th>
</tr>
</thead>
<tbody>
<tr>
<td class="org-left">GENERIC</td>
<td class="org-left">G</td>
<td class="org-left">Generic entities that does not fit other types yet or are temporarily only</td>
</tr>

<tr>
<td class="org-left">DOMAIN</td>
<td class="org-left">D</td>
<td class="org-left">Every Domain is of this type</td>
</tr>

<tr>
<td class="org-left">MEDIA</td>
<td class="org-left">M</td>
<td class="org-left">Every media item belongs to this type, e.g. Pictures, Audio, Video</td>
</tr>

<tr>
<td class="org-left">ITEM</td>
<td class="org-left">I</td>
<td class="org-left">An Entity what is usually used in a collection, e.g. Questions in a test</td>
</tr>

<tr>
<td class="org-left">COLLECTION</td>
<td class="org-left">C</td>
<td class="org-left">A collection of items, as an Exam or a catalog</td>
</tr>

<tr>
<td class="org-left">TEXT</td>
<td class="org-left">T</td>
<td class="org-left">A text document</td>
</tr>

<tr>
<td class="org-left">HANDOUT</td>
<td class="org-left">H</td>
<td class="org-left">A published document</td>
</tr>

<tr>
<td class="org-left">OUTPUT</td>
<td class="org-left">O</td>
<td class="org-left">The output of a computation</td>
</tr>

<tr>
<td class="org-left">RECORD</td>
<td class="org-left">R</td>
<td class="org-left">Record type data, as bibliography entries</td>
</tr>

<tr>
<td class="org-left">SESSION</td>
<td class="org-left">S</td>
<td class="org-left">A unique session, e.g. managed by a session manager</td>
</tr>

<tr>
<td class="org-left">USER</td>
<td class="org-left">U</td>
<td class="org-left">User objects</td>
</tr>

<tr>
<td class="org-left">CONFIG</td>
<td class="org-left">F</td>
<td class="org-left">CONFIG files that need to be tracked over time and state</td>
</tr>

<tr>
<td class="org-left">EVENT</td>
<td class="org-left">E</td>
<td class="org-left">Events that have to be tracked over time, as status messages or orders</td>
</tr>

<tr>
<td class="org-left">ATTESTATION</td>
<td class="org-left">X</td>
<td class="org-left">Entities that attest a formal technical (not human) check e.g. Signatures</td>
</tr>
</tbody>
</table>

<table border="2" cellspacing="0" cellpadding="6" rules="groups" frame="hsides">


<colgroup>
<col  class="org-left" />

<col  class="org-left" />

<col  class="org-left" />
</colgroup>
<thead>
<tr>
<th scope="col" class="org-left">EntityState</th>
<th scope="col" class="org-left">Code</th>
<th scope="col" class="org-left">Meaning</th>
</tr>
</thead>
<tbody>
<tr>
<td class="org-left">DRAFT</td>
<td class="org-left">D</td>
<td class="org-left">Work in progress</td>
</tr>

<tr>
<td class="org-left">APPROVED</td>
<td class="org-left">A</td>
<td class="org-left">Reviewed</td>
</tr>

<tr>
<td class="org-left">APPROVED<sub>AND</sub><sub>SIGNED</sub></td>
<td class="org-left">S</td>
<td class="org-left">Signed version</td>
</tr>

<tr>
<td class="org-left">PUBLISHED</td>
<td class="org-left">P</td>
<td class="org-left">Publicly released</td>
</tr>

<tr>
<td class="org-left">OBSOLETE</td>
<td class="org-left">O</td>
<td class="org-left">Deprecated</td>
</tr>
</tbody>
</table>


<a id="org659c733"></a>

# Design Notes

-   **Hash Stability:** Only domain, entity type, and content text influence the hash.
    This ensures consistent prefixes across state changes.
-   **State-Dependent Signatures:** Each lifecycle stage has its own signature seed.
    Modifying a file without re-signing invalidates integrity.
-   **Obsolescence Threshold:** Version numbers above 900 trigger warnings;
    beyond 999 are considered obsolete.
-   **Clone Lineages:** Cloning an entity resets versioning but preserves metadata lineage.


<a id="orga67f3a6"></a>

# Dependencies

-   Python 3.11+
-   Standard library only (`hashlib`, `json`, `datetime`, `enum`, `dataclasses`)

No external packages. Fully compatible with **Guix**, **air-gapped** deployments, and **reproducible builds**.


<a id="org10ab165"></a>

# Integration

\`flexoentity\` is imported by higher-level modules such as:

-   **Flex-O-Grader** → manages question catalogs and exam bundles
-   **Flex-O-Vault** → provides persistent media storage with metadata integrity
-   **Flex-O-Drill** → uses versioned entities for training simulations

All share the same identity and versioning logic — ensuring that
**what was approved, signed, and published remains provably authentic.**


<a id="org48ba119"></a>

# License

MIT License 2025
Part of the **Flex-O family** by Flex-O-Dyne GmbH 
Designed for reproducible, audit-ready, human-centered software.

