"""
id_factory.py — Central Flex-O ID generator and versioning control (hardened).

Improvements:
- BLAKE2s hashing (modern, fast, stdlib)
- 6 hex-digit hash (≈16.7M combinations)
- UTC-based dates for consistency
- Collision disambiguator (-A, -B, ...)
- Canonical seed and content fingerprint helpers
"""

import logging
from datetime import datetime, timezone
import hashlib
import itertools
import json

logger = logging.getLogger(__name__)
# ──────────────────────────────────────────────────────────────────────────────
#  Canonicalization helpers
# ──────────────────────────────────────────────────────────────────────────────


def canonical_seed(obj) -> str:
    """
    Deterministically flatten an entity's core data into a string
    for hashing and deduplication.
    """
    if isinstance(obj, str):
        text = " ".join(obj.lower().split())
        return text
    if isinstance(obj, dict):
        return json.dumps(obj, sort_keys=True, separators=(",", ":"))
    if hasattr(obj, "__dict__"):
        return canonical_seed(obj.__dict__)
    return str(obj)

# ──────────────────────────────────────────────────────────────────────────────
#  ID Factory
# ──────────────────────────────────────────────────────────────────────────────


class FlexOID:
    MAX_VERSION = 999
    WARN_THRESHOLD = 900

    # keep in-memory registry for same-session collisions (optional)
    _seen_hashes = set()

    @classmethod
    def from_oid_and_version(cls, oid, version: int):
        if not (1 <= version <= cls.MAX_VERSION):
            raise ValueError(f"Version {version} out of bounds (1..{cls.MAX_VERSION}).")
        return FlexOID(f"{oid.prefix}@{version:03d}{oid.state_code}", oid.signature)

    def __init__(self, flexo_id: str, signature: str):
        self.flexo_id = flexo_id
        self.signature = signature

    def __eq__(self, other):
        if not isinstance(other, FlexOID):
            return NotImplemented
        return self.flexo_id == other.flexo_id and self.signature == other.signature

    def __lt__(self, other):
        return self.version < other.version

    def __hash__(self):
        return hash((self.flexo_id, self.signature))

    @staticmethod
    def _blake_hash(text: str) -> str:
        """Return a 6-hex BLAKE2s digest."""
        return hashlib.blake2s(text.encode("utf-8"), digest_size=3).hexdigest().upper()  # 3 bytes → 6 hex

    @staticmethod
    def _ensure_unique(hash_part: str, version: int | None = None) -> str:
        """Append disambiguator if hash was already seen this session."""
        candidate = f"{hash_part}-{version}" if version is not None else hash_part
        if candidate not in FlexOID._seen_hashes:
            FlexOID._seen_hashes.add(candidate)
            return candidate
        # fallback only if truly same hash+version (should never happen)
        for suffix in itertools.chain(map(chr, range(65, 91)), range(1, 100)):
            alt = f"{candidate}-{suffix}"
            if alt not in FlexOID._seen_hashes:
                FlexOID._seen_hashes.add(alt)
                return alt
        raise RuntimeError("Too many collisions; adjust hash length or logic.")

    # ──────────────────────────────────────────────────────────────────────────
    @staticmethod
    def generate(domain: str, etype: str, estate: str, text: str, version: int = 1):
        """
        Generate a new, versioned, and state-aware Flex-O ID.

        This is the primary constructor for new entities. It combines the domain,
        entity type, creation date, unique hash (derived from the canonicalized
        text and lifecycle state), version number, and state suffix into a
        deterministic but unique identifier string.

        Format:
        <domain>-<etype><date>-<hash>@<version><state>
        Example:
        AF-Q251019-9B3E2@001A

        Parameters
        ----------
        domain : str
        Two-to-six letter domain or organizational prefix (e.g. "AF").
        etype : str
        One-to-three letter entity type identifier (e.g. "RQ" for Question).
        estate : str
        Lifecycle state whose one-letter code is appended to the version.
        text : str
        Entity-specific text seed (used to derive a stable hash).
        version : int, optional
        Starting version number (default is 1).

        Returns
        -------
        FlexOID
        A new Flex-O ID object with unique hash and digital signature.

        Notes
        -----
        - Changing the text or state will produce a new hash and signature.
        - Versions start at 1 and increment through `next_version()`.
        """

        if not (1 <= version <= FlexOID.MAX_VERSION):
            raise ValueError(f"Version {version} exceeds limit; mark obsolete.")

        date_part = datetime.now(timezone.utc).strftime("%y%m%d")

        # include state in the seed so hash & signature depend on lifecycle state
        seed = canonical_seed(f"{text}:{estate}")

        base_hash = FlexOID._blake_hash(seed)
        unique_hash = FlexOID._ensure_unique(base_hash, version)

        ver_part = f"{version:03d}{estate}"
        flexo_id_str = f"{domain}-{etype}{date_part}-{unique_hash}@{ver_part}"

        # short signature derived from state-aware seed
        signature = hashlib.blake2s(seed.encode("utf-8"), digest_size=8).hexdigest().upper()

        return FlexOID(flexo_id_str, signature)

    @property
    def state_code(self):
        part = self.flexo_id.rsplit("@", 1)[-1]

        if not (part and part[-1].isalpha()):
            raise ValueError(f"Invalid Flex-O ID format: {self.flexo_id}")
        return part[-1]

    @property
    def version(self) -> int:
        try:
            ver_state = self.flexo_id.rsplit("@", 1)[-1]
            return int(ver_state[:-1])  # drop state suffix
        except (ValueError, IndexError):
            return 1

    @property
    def prefix(self) -> str:
        return self.flexo_id.rsplit("@", 1)[0]
    # ──────────────────────────────────────────────────────────────────────────

    @classmethod
    def next_version(cls, oid) -> str:
        """
        Create the next version in the same ID lineage.

        Increments the version counter of an existing FlexOID while preserving
        its prefix and digital signature. Used when an entity transitions to a
        new revision within the same lifecycle (e.g., minor updates or approvals).

        Parameters
        ----------
        oid : FlexOID
        The existing ID whose version is to be incremented.

        Returns
        -------
        FlexOID
        A new Flex-O ID with the same prefix and signature, but version +1.

        Raises
        ------
        RuntimeError
        If the maximum allowed version (`MAX_VERSION`) is exceeded.

        Notes
        -----
        - The signature remains unchanged since the entity lineage is continuous.
        - Warnings are logged when the version approaches obsolescence.
        """
        new_ver = oid.version + 1

        if new_ver > cls.WARN_THRESHOLD and new_ver < cls.MAX_VERSION:
            logger.warning(f"{oid} approaching obsolescence ({new_ver}/999).")
        if new_ver > cls.MAX_VERSION:
            raise RuntimeError(f"{oid} exceeded {cls.MAX_VERSION}; mark obsolete.")

        new_id = f"{oid.prefix}@{new_ver:03d}{oid.state_code}"
        return cls(new_id, oid.signature)

    # ──────────────────────────────────────────────────────────────────────────
    @staticmethod
    def clone_new_base(domain: str, etype: str, estate: str, text: str):
        """
        Start a new Flex-O ID lineage for a derived or duplicated entity.

        This helper creates a completely new base ID (version 1) using the given
        parameters, instead of incrementing an existing version chain. It is used
        when an entity is copied, forked, or conceptually replaced by a new one.

        Returns
        -------
        FlexOID
        A new base ID starting at version 1, unrelated to the original lineage.

        Notes
        -----
        - Equivalent to calling `generate(..., version=1)` explicitly.
        - Used when creating "clones" or "variants" that should not share version history.
        """
        return FlexOID.generate(domain, etype, estate, text, version=1)

    def __str__(self):
        return self.flexo_id

    def __repr__(self):
        return f"<FlexOID {self.flexo_id} sig={self.signature[:8]}…>"

    @classmethod
    def from_str(cls, id_str: str):
        # reconstruct without a known signature
        return cls(id_str, signature="")
