"""
versioned_entity.py — Shared base class for all Flex-O entities.
Integrates with hardened id_factory.py and content fingerprint tracking.
"""
import json
from enum import Enum, auto
from dataclasses import dataclass, field
from datetime import datetime
from typing import Optional
from abc import ABC

from id_factory import FlexOID


# ──────────────────────────────────────────────────────────────────────────────
#  Entity Types
# ──────────────────────────────────────────────────────────────────────────────


class EntityType(Enum):
    QUESTION = auto()
    MEDIA = auto()
    CATALOG = auto()
    EXAM = auto()
    DATABASE = auto()
    CERTIFICATE = auto()

    def short(self) -> str:
        mapping = {
            EntityType.QUESTION: "Q",
            EntityType.CATALOG:  "CAT",
            EntityType.EXAM:     "EX",
            EntityType.DATABASE: "DB",
            EntityType.CERTIFICATE: "CERT"
        }
        return mapping[self]

# ──────────────────────────────────────────────────────────────────────────────
#  States
# ──────────────────────────────────────────────────────────────────────────────


class EntityState(Enum):
    DRAFT = auto()
    APPROVED = auto()
    APPROVED_AND_SIGNED = auto()
    PUBLISHED = auto()
    OBSOLETE = auto()

    def short(self) -> str:
        """
        Return a one-letter abbreviation for the state, used in Flex-O IDs.
        """
        mapping = {
            EntityState.DRAFT: "D",
            EntityState.APPROVED: "A",
            EntityState.APPROVED_AND_SIGNED: "S",
            EntityState.OBSOLETE: "O",
        }
        return mapping[self]

    @classmethod
    def from_short(cls, char: str):
        """
        Inverse of .short(): restore the EntityState from its one-letter code.
        """
        reverse = {
            "D": cls.DRAFT,
            "A": cls.APPROVED,
            "S": cls.APPROVED_AND_SIGNED,
            "O": cls.OBSOLETE,
        }
        try:
            return reverse[char.upper()]
        except KeyError:
            raise ValueError(f"Unknown state abbreviation: {char}")

    def __str__(self):
        return self.value


@dataclass
class FlexoEntity(ABC):
    domain: str
    etype: EntityType
    text_seed: str
    state: EntityState = EntityState.DRAFT
    flexo_id: FlexOID = field(init=False)
    created_at: datetime = field(default_factory=datetime.utcnow)
    updated_at: Optional[datetime] = None

    # ───────────────────────────────────────────────────────────────

    def __post_init__(self):
        """Generate ID and content fingerprint."""
        self.flexo_id = FlexOID.generate(self.domain, self.etype.short(), self.state.short(), self.text_seed, 1)

    def __str__(self):
        return (
            f"{self.etype.name}({self.flexo_id}, {self.state.name}, "
            f"sig={self.flexo_id.signature}..., v{self.version})"
        )

    def to_dict(self):
        return {
            "domain": self.domain,
            "etype": self.etype.name,
            "text_seed": self.text_seed,
            "state": self.state.name,
            "version": self.version,
            "flexo_id": str(self.flexo_id),
            "signature": self.flexo_id.signature,
            "created_at": self.created_at.isoformat(),
            "updated_at": self.updated_at.isoformat() if self.updated_at else None,
        }

    @classmethod
    def from_dict(cls, data):
        obj = cls(
            data["domain"],
            EntityType[data["etype"]],
            data["text_seed"],
            EntityState[data["state"]],
            data["version"],
        )
        obj.flexo_id = FlexOID(data["flexo_id"], data.get("signature", ""))
        if data.get("updated_at"):
            obj.updated_at = datetime.fromisoformat(data["updated_at"])
        return obj

    def to_json(self, *, indent: int | None = None) -> str:
        """Serialize entity (and its FlexOID) into JSON."""
        return json.dumps(self.to_dict(), indent=indent, ensure_ascii=False)

    @classmethod
    def from_json(cls, data_str: str) -> "FlexoEntity":
        """Deserialize from a JSON string."""
        data = json.loads(data_str)
        return cls.from_dict(data)

    # ───────────────────────────────────────────────────────────────
    def _update_fingerprint(self) -> bool:
        """Recalculate fingerprint and return True if content changed."""
        # extract version from current flexo_id
        new_oid = FlexOID.generate(self.domain, self.etype, self.text_seed, self.flexo_id.version)
        if new_oid.signature != self.flexo_id.signature:
            self.flexo_id = new_oid
            return True
        return False

    # ───────────────────────────────────────────────────────────────
    def _transition(self, target_state: EntityState):
        """Internal helper for state transitions with version and fingerprint checks."""
        if target_state == EntityState.OBSOLETE:
            self.state = target_state
            return

        # Check if version should bump
        if FlexOID.should_version(self.etype, target_state):
            if self._update_fingerprint():
                self.flexo_id = FlexOID.next_version(self.flexo_id)

        self.state = target_state
        self.updated_at = datetime.utcnow()

    # ───────────────────────────────────────────────────────────────
    # Lifecycle transitions
    # ───────────────────────────────────────────────────────────────

    @property
    def version(self) -> int:
        """Extract numeric version from the FlexO-ID string."""
        try:
            return int(str(self.flexo_id).rsplit("@", 1)[-1])
        except (ValueError, AttributeError):
            return 1  # fallback if malformed or missing

    def bump_version(self):
        """Increment version number on the ID."""
        self.flexo_id = FlexOID.next_version(self.flexo_id)
        self.updated_at = datetime.utcnow()

    def approve(self):
        """
        Move from DRAFT to APPROVED state.
        Draft entities receive a new permanent FlexOID.
        """
        if self.state == EntityState.DRAFT:
            # Generate a brand new permanent ID
            new_fid = FlexOID.generate(
                self.domain,
                self.etype,
                self.text_seed,
                draft=False
            )
            self.previous_id = self.flexo_id  # optional: keep audit trail
            self.flexo_id = new_fid

            self.state = EntityState.APPROVED
            self.updated_at = datetime.utcnow()
            return self
        raise ValueError("Only drafts can be approved")

    def sign(self):
        # FIXME: We need to define clear conditions, when resigning is neccessary and allowed
        #        if self.state == EntityState.APPROVED:
        #            self._transition(EntityState.APPROVED_AND_SIGNED)
        #            self.bump_version()
        self._transition(EntityState.APPROVED_AND_SIGNED)
        self.bump_version()

    def publish(self):
        if self.state == EntityState.APPROVED_AND_SIGNED:
            self._transition(EntityState.PUBLISHED)

    def obsolete(self):
        if self.state != EntityState.OBSOLETE:
            self._transition(EntityState.OBSOLETE)

    def clone_new_base(self):
        """Start new lineage when obsolete."""
        self.flexo_id = FlexOID.clone_new_base(self.domain, self.etype, self.text_seed)
        self.state = EntityState.DRAFT
        self.updated_at = datetime.utcnow()

    # ───────────────────────────────────────────────────────────────
    def modify_content(self, new_text_seed: str):
        """
        Update content and detect whether fingerprint changes.
        Does not increment version automatically; transitions handle that.
        """
        self.text_seed = new_text_seed.strip()
        if self._update_fingerprint():
            # mark content changed, but version will only bump on next approval/sign/publish
            self.updated_at = datetime.utcnow()

    @staticmethod
    def verify_integrity(entity) -> bool:
        """
        Verify if the stored fingerprint matches recalculated fingerprint.
        Returns True if intact, False if tampered or corrupted.
        """
        recalculated = FlexOID.generate(entity.domain, entity.etype, entity.text_seed, entity.version)
        return recalculated.signature == entity.flexo_id.signature

    def allowed_transitions(self) -> list[str]:
        """Return a list of possible next state names (for GUI use)."""
        match self.state:
            case EntityState.DRAFT:
                return ["APPROVED"]
            case EntityState.APPROVED:
                return ["APPROVED_AND_SIGNED", "OBSOLETE"]
            case EntityState.APPROVED_AND_SIGNED:
                return ["PUBLISHED", "OBSOLETE"]
            case EntityState.PUBLISHED:
                return ["OBSOLETE"]
            case EntityState.OBSOLETE:
                return []
            case _:
                return []
