import sqlite3
import json
from typing import Optional, Iterable

from .persistance_backend import PersistanceBackend


class SQLiteEntityBackend(PersistanceBackend):
    """
    Generic SQLite persistence layer for FlexOEntity subclasses.
    
    Subclasses must define:
      - TABLE_NAME        (e.g. "users")
      - ENTITY_CLASS      (e.g. FlexoUser)
      - INDEXED_FIELDS    (list of field names to index)
    """

    TABLE_NAME = None
    ENTITY_CLASS = None
    INDEXED_FIELDS: list[str] = []

    def __init__(self, db_path: str):
        if self.TABLE_NAME is None:
            raise ValueError("Subclasses must define TABLE_NAME")
        if self.ENTITY_CLASS is None:
            raise ValueError("Subclasses must define ENTITY_CLASS")

        self.db_path = db_path
        self.conn = sqlite3.connect(db_path)
        self.conn.row_factory = sqlite3.Row
        self._init_schema()

    # -----------------------------------------------------------
    # Schema
    # -----------------------------------------------------------
    def _init_schema(self):
        # Base schema: store full entity JSON and the FlexOID
        columns = [
            "flexo_id TEXT PRIMARY KEY",
            "json TEXT NOT NULL",
        ]

        # Add indexed metadata columns
        for field in self.INDEXED_FIELDS:
            columns.append(f"{field} TEXT")

        colspec = ", ".join(columns)

        self.conn.execute(
            f"CREATE TABLE IF NOT EXISTS {self.TABLE_NAME} ({colspec})"
        )

        # Create indices on metadata fields
        for field in self.INDEXED_FIELDS:
            self.conn.execute(
                f"CREATE INDEX IF NOT EXISTS idx_{self.TABLE_NAME}_{field} "
                f"ON {self.TABLE_NAME}({field})"
            )

        self.conn.commit()

    # -----------------------------------------------------------
    # CRUD operations
    # -----------------------------------------------------------
    def add(self, obj):
        data = json.dumps(obj.to_dict(), ensure_ascii=False)

        # Collect row data dynamically
        fields = ["flexo_id", "json"] + self.INDEXED_FIELDS
        placeholders = ", ".join("?" for _ in fields)
        sql = (
            f"INSERT INTO {self.TABLE_NAME} ({', '.join(fields)}) "
            f"VALUES ({placeholders})"
        )

        params = [obj.flexo_id.value, data]
        params.extend(getattr(obj, f) for f in self.INDEXED_FIELDS)

        self.conn.execute(sql, params)
        self.conn.commit()

    def update(self, obj):
        data = json.dumps(obj.to_dict(), ensure_ascii=False)
        set_parts = ["json = ?"] + [f"{f} = ?" for f in self.INDEXED_FIELDS]

        sql = (
            f"UPDATE {self.TABLE_NAME} SET "
            f"{', '.join(set_parts)} "
            f"WHERE flexo_id = ?"
        )

        params = [data]
        params.extend(getattr(obj, f) for f in self.INDEXED_FIELDS)
        params.append(obj.flexo_id.value)

        self.conn.execute(sql, params)
        self.conn.commit()

    def delete(self, flexo_id: str):
        self.conn.execute(
            f"DELETE FROM {self.TABLE_NAME} WHERE flexo_id = ?",
            (flexo_id,),
        )
        self.conn.commit()

    # -----------------------------------------------------------
    # Retrieval
    # -----------------------------------------------------------
    def get(self, flexo_id: str):
        row = self.conn.execute(
            f"SELECT json FROM {self.TABLE_NAME} WHERE flexo_id = ?",
            (flexo_id,),
        ).fetchone()
        if not row:
            return None
        return self.ENTITY_CLASS.from_dict(json.loads(row["json"]))

    def all(self):
        rows = self.conn.execute(
            f"SELECT json FROM {self.TABLE_NAME}"
        ).fetchall()
        return [
            self.ENTITY_CLASS.from_dict(json.loads(r["json"])) for r in rows
        ]

    # -----------------------------------------------------------
    # Dynamic finders: find_by_<field>
    # -----------------------------------------------------------
    def __getattr__(self, name):
        # e.g. find_by_username("foo")
        if name.startswith("find_by_"):
            field = name[len("find_by_"):]
            if field not in self.INDEXED_FIELDS:
                raise AttributeError(f"No such indexed field: {field}")

            def finder(value):
                rows = self.conn.execute(
                    f"SELECT json FROM {self.TABLE_NAME} WHERE {field} = ?",
                    (value,),
                )
                for row in rows:
                    yield self.ENTITY_CLASS.from_dict(json.loads(row["json"]))

            return finder

        raise AttributeError(name)
