"""
flexoentity.collection
----------------------

A minimal collection for FlexOEntities.

- Default key = entity.flexo_id
- Optional override key=... for special cases (e.g., domains)
- Smalltalk-like protocol preserved
- Dict-based: fast lookup
- Backwards compatible API
"""
from abc import abstractmethod
from .flexo_entity import FlexoEntity
from .id_factory import FlexOID


class FlexoCollection:
    """
    A minimal collection of FlexOEntities, keyed by FlexOID by default.

    Examples:
        coll = FlexoCollection()
        coll.add(entity)                      # uses entity.flexo_id as key
        coll.add(domain, key=domain.domain_id) # override key if needed
        ent = coll.get(some_id)
        for e in coll:
            ...
    """

    def __init__(self, items=None):
        # internal dict-based storage
        self._items = {}

        if items:
            for it in items:
                self.add(it)

    # ------------------------------------------------------------------
    # Core API (unchanged externally, enhanced internally)
    # ------------------------------------------------------------------

    def add(self, entity: FlexoEntity, key=None):
        """
        Add or replace an entity.

        - If key is None → use entity.flexo_id
        - If key is given → use caller-provided key
        """
        if key is None:
            try:
                key = entity.flexo_id
            except AttributeError:
                raise TypeError(
                    "Items must have .flexo_id or explicitly use add(entity, key=...)."
                )

        self._items[key] = entity  # overwrite is intentional & preserved

    def remove(self, oid: FlexOID):
        """Remove an entity by ID, ignoring if missing (API preserved)."""
        self._items.pop(oid, None)

    def clear(self):
        self._items = {}

    def get(self, oid: FlexOID):
        """Return the entity or None."""
        return self._items.get(oid)

    def __contains__(self, oid: FlexOID):
        """Check if an ID exists."""
        return oid in self._items

    def __len__(self) -> int:
        return len(self._items)

    def __iter__(self):
        return iter(self._items.values())

    # ------------------------------------------------------------------
    # Additional public helpers (unchanged)
    # ------------------------------------------------------------------
    def entities(self):
        """Return all entities as a list."""
        return list(self._items.values())

    def ids(self):
        """Return all stored keys."""
        return list(self._items.keys())

    # ------------------------------------------------------------------
    # Smalltalk-inspired interface (unchanged)
    # ------------------------------------------------------------------

    def at(self, oid):
        """Smalltalk-style accessor (raises if missing)."""
        return self._items[oid]

    def at_put(self, oid, entity):
        """Smalltalk-style setter (kept)."""
        self._items[oid] = entity

    def includes_id(self, oid: FlexOID):
        """Smalltalk-style membership test."""
        return oid in self._items

    def do(self, block):
        """Smalltalk-style #do: iteration."""
        for value in self._items.values():
            block(value)

    # ------------------------------------------------------------------
    # Serialization helpers (unchanged)
    # ------------------------------------------------------------------

    def serialize(self):
        """Serialize all entities to list of dicts."""
        return [e.to_dict() for e in self.entities()]

    @classmethod
    @abstractmethod
    def deserialize(cls, data):
        """
        Deserialize a list of dicts into a FlexoCollection.

        Uses from_dict() to dispatch to the correct subclass.
        """
        pass

    # ------------------------------------------------------------------

    def __repr__(self) -> str:
        return f"<FlexoCollection size={len(self._items)}>"
