from .persistance_backend import PersistanceBackend
from .flexo_entity import FlexoEntity
from .in_memory_backend import InMemoryBackend

class CompositeBackend(PersistanceBackend):
    """
    A backend that wraps multiple real backends.

    - Reads always come from the first backend (e.g., InMemoryBackend).
    - Writes propagate to all backends.
    """

    def __init__(self, entity_class, *backends):
        if not issubclass(entity_class, FlexoEntity):
            raise TypeError("entity_class must be a subclass of FlexOEntity")

        self.entity_class = entity_class

        # Default: create an in-memory backend as the primary backend
        if not backends:
            backends = (InMemoryBackend(entity_class),)

        # Validate all backends
        for b in backends:
            if getattr(b, "entity_class", None) != entity_class:
                raise TypeError(
                    f"Backend {b} does not match entity_class={entity_class.__name__}"
                )

        self.backends = list(backends)

    @property
    def primary(self):
        """The backend used for all read operations."""
        return self.backends[0]

    def add_backend(self, backend, clear=False):
        """
        Append an additional backend.
        If clear=True, backend is wiped before syncing.
        """
        if backend.entity_class != self.entity_class:
            raise TypeError("Backend entity_class mismatch")

        if clear:
            backend.delete_all()

        # Sync current data into backend
        for entity in self.primary.load_all():
            backend.save(entity)

        self.backends.append(backend)

    def remove_backend(self, backend):
        """
        Remove a backend. Primary backend cannot be removed.
        """
        if backend is self.primary:
            raise ValueError("Cannot remove the primary backend")
        self.backends.remove(backend)           
    # ---------------------------------------------------------
    # Write operations propagate to *all* backends
    # ---------------------------------------------------------

    def save(self, entity):
        for b in self.backends:
            b.save(entity)

    def update(self, entity):
        for b in self.backends:
            b.update(entity)

    def delete(self, flexo_id: str):
        for b in self.backends:
            b.delete(flexo_id)

    # ---------------------------------------------------------
    # Read operations use only the *primary* backend
    # ---------------------------------------------------------

    def load(self, flexo_id: str):
        return self.primary.load(flexo_id)

    def load_all(self):
        return self.primary.load_all()

    # ---------------------------------------------------------
    # Optional: flush from primary backend to all others
    # ---------------------------------------------------------

    def sync_all(self):
        """
        Push all data from the primary backend to the other backends.
        Useful if secondary backends were empty initially.
        """
        primary = self.backends[0]
        for entity in primary.load_all():
            for b in self.backends[1:]:
                b.save(entity)

    def clear(self):
        for b in self.backends:
            if hasattr(b, "clear"):
                b.clear()

    # ---------------------------------------------------------

    def __repr__(self):
        names = ", ".join(b.__class__.__name__ for b in self.backends)
        return f"<CompositeBackend [{names}]>"
