import json
from .persistance_backend import PersistanceBackend


class JsonFileBackend(PersistanceBackend):
    def __init__(self, path: str):
        self.path = path
        self.data = []
        self._load_file()

    def _load_file(self):
        try:
            with open(self.path, "r", encoding="utf-8") as f:
                self.data = json.load(f)
        except FileNotFoundError:
            self.data = []

    def save(self, entity_dict):
        self.data.append(entity_dict)
        self._flush()

    def update(self, entity_dict):
        fid = entity_dict["flexo_id"]
        self.data = [d for d in self.data if d["flexo_id"] != fid]
        self.data.append(entity_dict)
        self._flush()

    def delete(self, flexo_id):
        self.data = [d for d in self.data if d["flexo_id"] != flexo_id]
        self._flush()

    def load(self, flexo_id):
        for d in self.data:
            if d["flexo_id"] == flexo_id:
                return d
        return None

    def load_all(self):
        return self.data

    def _flush(self):
        with open(self.path, "w", encoding="utf-8") as f:
            json.dump(self.data, f, ensure_ascii=False, indent=2)
