import json
from .flexo_entity import FlexoEntity
from .persistance_backend import PersistanceBackend


class InMemoryBackend(PersistanceBackend):
    """
    Persistence backend for a *typed* collection of FlexOEntities.

    - Stores dicts internally (no entities)
    - Reconstructs entities using ENTITY_CLASS.from_dict() on load
    - Accepts entity_class injected at construction time
    """

    def __init__(self, entity_class, storage=None):
        if not issubclass(entity_class, FlexoEntity):
            raise TypeError("entity_class must be a subclass of FlexOEntity")

        self.entity_class = entity_class
        self._storage = storage if storage is not None else {}

    # ---------------------------------------------------------
    # Core persistence API (dict-based internally)
    # ---------------------------------------------------------

    def save(self, entity):
        """Store serialized entity dict by flexo_id."""
        data = entity.to_dict()
        flexo_id = data["meta"]["flexo_id"]
        self._storage[flexo_id] = data

    def update(self, entity):
        self.save(entity)

    def delete(self, flexo_id: str):
        self._storage.pop(flexo_id, None)

    def load(self, flexo_id: str):
        """Return an entity instance or None."""
        d = self._storage.get(flexo_id)
        return None if d is None else self.entity_class.from_dict(d)

    def load_all(self):
        """Return a list of entity instances."""
        return [
            self.entity_class.from_dict(d)
            for d in self._storage.values()
        ]

    def clear(self):
        self._storage.clear()

    # ---------------------------------------------------------
    # Optional: JSON file persistence
    # ---------------------------------------------------------

    def save_to_file(self, path):
        with open(path, "w", encoding="utf-8") as f:
            json.dump(
                list(self._storage.values()),
                f,
                ensure_ascii=False,
                indent=2
            )

    def load_from_file(self, path):
        with open(path, "r", encoding="utf-8") as f:
            data = json.load(f)
        # Replace storage with file content
        self._storage = {d["meta"]["flexo_id"]: d for d in data}
