from .flexo_entity import FlexoEntity
from .typed_collection import TypedCollection


class EntityManager:
    """
    Backend-agnostic manager for any FlexOEntity subclass.

    Responsibilities:
      - enforce ENTITY_CLASS (e.g., FlexoUser, Domain, Question)
      - convert entities <-> dicts
      - delegate persistence to backend
      - provide a consistent CRUD API for GUIs and CLI
      - avoid backend-specific code in subclasses
    """

    ENTITY_CLASS = None  # must be overridden by subclasses

    def __init__(self, backend):
        if self.ENTITY_CLASS is None:
            raise ValueError("Subclasses must define ENTITY_CLASS")

        if not issubclass(self.ENTITY_CLASS, FlexoEntity):
            raise TypeError("ENTITY_CLASS must be a subclass of FlexOEntity")
        self.backend = backend

    # ------------------------------------------------------------------
    # CRUD operations
    # ------------------------------------------------------------------

    def add(self, entity):
        """Insert a new entity."""
        self._ensure_type(entity)
        self.backend.save(entity)

    def update(self, entity):
        """Replace an existing entity (same flexo_id)."""
        self._ensure_type(entity)
        self.backend.update(entity)

    def delete(self, flexo_id: str):
        """Remove entity by string flexo_id."""
        self.backend.delete(flexo_id)

    # ------------------------------------------------------------------
    # Retrieval
    # ------------------------------------------------------------------

    def get(self, flexo_id: str):
        """
        Load entity by flexo_id str.
        Returns ENTITY_CLASS instance or None.
        """
        return self.backend.load(flexo_id)

    def all(self):
        """Load all entities as a list of instances."""
        return self.backend.load_all()
    # ------------------------------------------------------------------
    # Helpers
    # ------------------------------------------------------------------

    def exists(self, flexo_id: str) -> bool:
        return self.get(flexo_id) is not None

    def count(self) -> int:
        return len(self.backend.load_all())

    def add_or_update(self, entity):
        """
        Convenience for GUIs:
        Insert or overwrite based on whether flexo_id exists.
        """
        if self.exists(entity.flexo_id.value):
            self.update(entity)
        else:
            self.add(entity)

    def as_collection(self):
        return TypedCollection(self.ENTITY_CLASS, items=self.backend.load_all())

    # ------------------------------------------------------------------
    # Internal
    # ------------------------------------------------------------------

    def _ensure_type(self, entity):
        if not isinstance(entity, self.ENTITY_CLASS):
            raise TypeError(
                f"Expected {self.ENTITY_CLASS.__name__}, "
                f"got {type(entity).__name__}"
            )

    # ------------------------------------------------------------------

    def __repr__(self):
        return f"<FlexoEntityManager for {self.ENTITY_CLASS.__name__}>"
