"""
flexo_entity.py - I represent the mutable counterpart of the immutable FlexOID.

I exist to connect an entity’s immutable identity with its mutable provenance
and content.  My FlexOID encodes what an entity *is* - its domain, type,
version, and lifecycle state - while I record *who* created it, *who* owns it
now, and *where* it came from.

Structure
──────────
- My `flexo_id` is the canonical source of truth.
  It deterministically encodes:
    - entity type (single letter, e.g. I, C, M)
    - date + hash (prefix stability)
    - version     (numeric counter)
    - state       (single capital letter)

- My derived properties (state, entity_type, version) read from that ID.
  They are never stored separately, so I cannot become inconsistent.

- My attributes origin, originator_id, and owner_id express authorship
  and workflow context:
    - origin links to the FlexOID of the entity I was derived from
    - originator_id identifies the original creator (UUID)
    - owner_id identifies the current responsible user or process (UUID)

Design principles
──────────────────
I separate immutable identity from mutable context:

    ┌────────────────────────────┐
    │ FlexOID                    │
    │  – who I am                │
    │  – lifecycle state         │
    └────────────▲───────────────┘
                 │
    ┌────────────┴───────────────┐
    │ FlexoEntity (this module)  │
    │  – who created or owns me  │
    │  – where I came from       │
    │  – what content I hold     │
    └────────────────────────────┘

This means:
  - My identity never changes; only my content or ownership does.
  - My lifecycle transitions (approve, sign, publish) are represented by new
    FlexOIDs, not by mutating my own attributes.
  - My audit trail and access control live in the provenance layer, not the ID.

I am the living object behind a FlexOID — mutable, traceable, and accountable.
"""

import json
from uuid import UUID
from enum import Enum
from dataclasses import dataclass, field, replace
from typing import Optional
from pathlib import Path
from abc import ABC, abstractmethod
import hashlib


from flexoentity.id_factory import FlexOID
from flexoentity.logger import logger


SCHEMA_VERSION = "1.0"
SCHEMA_NAME = "flexograder-entity"

# ──────────────────────────────────────────────────────────────────────────────
#  Entity Types
# ──────────────────────────────────────────────────────────────────────────────


class EntityType(Enum):
    """
    I map a fixed number of entity types to a single unique letter,
    that structures different types
    """
    GENERIC = "G"
    DOMAIN = "D"
    MEDIA = "M"
    ITEM = "I"
    CATALOG = "C"
    TEXT = "T"
    HANDOUT = "H"
    OUTPUT = "O"
    RECORD = "R"
    SESSION = "S"
    USER = "U"
    CONFIG = "F"
    EVENT = "E"
    ATTESTATION = "X"

    @classmethod
    def from_letter(cls, a_letter):
        """I am a convenience constructor method for EntityType(a_letter)"""
        return cls(a_letter)

# ──────────────────────────────────────────────────────────────────────────────
#  States
# ──────────────────────────────────────────────────────────────────────────────


class EntityState(Enum):
    """I describe the life cycle states for all entity types"""
    DRAFT = "D"
    APPROVED = "A"
    APPROVED_AND_SIGNED = "S"
    PUBLISHED = "P"
    OBSOLETE = "O"

    def allowed_transitions(self) -> list[str]:
        """Return allowed state names for this state."""
        return TRANSITIONS[self]

    def __str__(self):
        return self.name


TRANSITIONS = {
    EntityState.DRAFT: [EntityState.APPROVED],
    EntityState.APPROVED: [EntityState.APPROVED_AND_SIGNED, EntityState.PUBLISHED,
                           EntityState.OBSOLETE],
    EntityState.APPROVED_AND_SIGNED: [EntityState.PUBLISHED, EntityState.OBSOLETE],
    EntityState.PUBLISHED: [EntityState.OBSOLETE],
    EntityState.OBSOLETE: [],
}


class FlexoEntity(ABC):
    """
    I represent a mutable entity identified by an immutable FlexOID.

    My flexo_id is the single source of truth for what I am:
      - it encodes my domain, entity type, version, and lifecycle state.
    I never store those values separately, but derive them on demand.

    I extend the raw identity with provenance and authorship data:
      - origin        -> the FlexOID of the entity I was derived from
      - originator_id -> the UUID of the first creator
      - owner_id      -> the UUID of the current responsible person or system

    I am designed to be traceable and reproducible:
      - My identity (flexo_id) never mutates.
      - My lifecycle transitions are represented by new FlexOIDs.
      - My provenance (origin, originator_id, owner_id) can change over time
        as the entity moves through workflows or ownership transfers.

    Behavior
    ─────────
    - state and entity_type are read-only views derived from my FlexOID.
    - to_dict() and from_dict() serialize me with human-readable fields
      while preserving the canonical FlexOID.
    - Subclasses (such as questions, media items, or catalogs) extend me with
      domain-specific content and validation, but not with new identity rules.
    - Subclasses have to define a class attribute ENTITY_TYPE to allow specific instance creation

    I am the living, editable layer that connects identity with accountability.
    """

    def with_new_owner(self, new_owner: UUID):
        """I return a clone of this entity with a different owner."""
        copy = replace(self)
        copy.owner_id = new_owner
        return copy

    def __init__(self, *, flexo_id, subtype=None, fingerprint=None,
                 origin=None, originator_id=None, owner_id=None):

        if flexo_id is None:
            raise ValueError("flexo_id must be provided")

        self.flexo_id = flexo_id
        self.subtype = subtype or self.__class__.__name__
        self.origin = origin
        self.originator_id = originator_id
        self.owner_id = owner_id
        self.fingerprint = fingerprint

    @staticmethod
    def canonicalize_content_dict(data) -> str:
        """
        Canonicalize structured content for fingerprinting.
        - Stable JSON representation
        - Sorted keys
        - No whitespace noise
        """
        return json.dumps(data, sort_keys=True, separators=(",", ":"))

    def __eq__(self, other):
        if not isinstance(other, FlexoEntity):
            return NotImplemented
        return self.flexo_id == other.flexo_id

    def __hash__(self):
        return hash(self.flexo_id)

    @property
    @abstractmethod
    def text_seed(self) -> str:
        """I provide a canonicalized text used for ID generation."""
        raise NotImplementedError("Subclasses must define text_seed property")

    @property
    def domain_id(self):
        """I return my domain_id derived from FlexOID"""
        return self.flexo_id.domain_id

    @property
    def state(self) -> EntityState:
        """I return the state derived from my FlexOID"""
        return EntityState(self.flexo_id.state_code)

    @property
    def entity_type(self) -> EntityType:
        """I return the entity type derived from my FlexOID"""
        return EntityType(self.flexo_id.entity_type)

    @classmethod
    @abstractmethod
    def default(cls):
        """I return a minimal valid instance of this entity (in DRAFT state)."""
        raise NotImplementedError("Subclasses must implement default()")


    @classmethod
    def with_domain_id(cls, domain_id: str, **kwargs):
        entity_type = getattr(cls, "ENTITY_TYPE", None)
        if not entity_type:
            raise ValueError(f"{cls.__name__} must define ENTITY_TYPE")

        oid = FlexOID.safe_generate(
            domain_id=domain_id,
            entity_type=entity_type.value,
            state=EntityState.DRAFT.value,
            text="",
            version=1,
        )

        obj = cls(flexo_id=oid, **kwargs)
        obj.fingerprint = obj._compute_fingerprint()
        return obj

    def __str__(self):
        return (
            f"{self.subtype}-{self.entity_type.name}({self.flexo_id}, {self.state.name}, "
            f"fingerprint={self.fingerprint}..., v{self.version})"
        )

    def meta_dict(self):
        return {
            "flexo_id": str(self.flexo_id),
            "subtype": self.subtype,
            "fingerprint": self.fingerprint,
            "origin": self.origin,
            "originator_id": str(self.originator_id),
            "owner_id": str(self.owner_id),
    }

    def to_dict(self):

        return {
            "meta": {
                "schema": {
                    "name": SCHEMA_NAME,
                    "version": SCHEMA_VERSION,
                },
                **self.meta_dict(),
            },
            "content": self._serialize_content(),
        }

    @classmethod
    def from_dict(cls, data: dict):
        meta = data["meta"]
        content = data["content"]

        flexo_id = FlexOID(meta["flexo_id"])

        obj = cls(
            flexo_id=flexo_id,
            subtype=meta.get("subtype"),
            fingerprint=meta.get("fingerprint"),
            origin=meta.get("origin"),
            originator_id=meta.get("originator_id"),
            owner_id=meta.get("owner_id"),
        )

        obj._deserialize_content(content)

        return obj

    def fork(
        self,
        *,
        domain_id: str | None = None,
        reset_state: bool = True,
        keep_origin: bool = True,
    ) -> "FlexoEntity":
        """
        Create a new independent entity derived from this one.
        """
        data = self.to_dict()
        meta = data["meta"]

        # Provenance
        if keep_origin:
            meta["origin"] = str(self.flexo_id)

        # Reset lifecycle
        if reset_state:
            meta["version"] = 1
            meta["state"] = EntityState.DRAFT.value

        # Generate new identity
        new_flexo_id = FlexOID.safe_generate(
            domain_id=domain_id or self.domain_id,
            entity_type=self.entity_type.value,
            state=meta["state"],
            text=self.text_seed,
            version=meta["version"],
        )

        meta["flexo_id"] = str(new_flexo_id)

        return self.__class__.from_dict(data)

    def to_json(self, indent: int = 2) -> str:
        """Return a JSON representation including all FlexO metadata."""
        return json.dumps(self.to_dict(), ensure_ascii=False, indent=indent)

    def to_json_file(self, path):
        Path(path).write_text(self.to_json(), encoding="utf-8")
        return path

    @classmethod
    def from_json(cls, data_str: str):
        """I create a new instance from a JSON string."""
        data = json.loads(data_str)
        return cls.from_dict(data)

    @classmethod
    def from_json_file(cls, filename: str):
        with open(filename, "r", encoding="utf-8") as f:
            data = json.load(f)
        return cls.from_dict(data)


    def _deserialize_content(self, content_dict):
        """Subclasses override this to restore real fields."""
        pass  # default: do nothing


    def _serialize_content(self):
        """
        Subclasses override to return all integrity-relevant fields.
        Should include:
        - all actual structured data
        - comment (if meaningful)
        - signature_data / certificate info (if present)
        """
        return {}

    def canonical_content(self) -> str:
        """Get canonical JSON string of content for fingerprinting."""
        return self.canonicalize_content_dict(self._serialize_content())

    def _compute_fingerprint(self) -> str:
        """I recompute the entity's content fingerprint."""
        return hashlib.blake2s(self.canonical_content().encode("utf-8"),
                               digest_size=8).hexdigest().upper()

    def _update_fingerprint(self) -> bool:
        new_fp = self._compute_fingerprint()
        changed = new_fp != self.fingerprint
        self.fingerprint = new_fp
        return changed

    # ───────────────────────────────────────────────────────────────
    # Lifecycle transitions
    # ───────────────────────────────────────────────────────────────

    @property
    def version(self) -> int:
        """I extract numeric version from the FlexO-ID string."""
        try:
            return int(str(self.flexo_id).rsplit("@", 1)[-1])
        except (ValueError, AttributeError):
            return 1  # fallback if malformed or missing

    def bump_version(self):
        """I increment the version number on the ID."""
        self.flexo_id = FlexOID.next_version(self.flexo_id)

    def lineage(self, repo):
        """I return full ancestry chain [origin → ... → self]."""
        chain = [self]
        current = self
        while current.origin and current.origin in repo:
            current = repo[current.origin]
            chain.insert(0, current)
        return chain

    def approve(self):
        """
        I move from DRAFT to APPROVED state.
        Draft entities receive a new permanent FlexOID with incremented version.
        """
        if self.state == EntityState.DRAFT:
            new_fid = FlexOID.safe_generate(domain_id=self.domain_id,
                                            entity_type=self.entity_type.value,
                                            state=EntityState.APPROVED.value,
                                            text=self.text_seed,
                                            version=self.version
                                            )
            self.origin = self.flexo_id  # optional: keep audit trail
            self.flexo_id = new_fid
            return self
        raise ValueError("Only drafts can be approved")

    def sign(self):
        """
        I mark entity as approved and signed without bumping version.
        Only changes the state letter in the FlexOID.
        FIXME: We need a signature here
        """
        if self.state != EntityState.APPROVED:
            raise ValueError("Only approved entities can be signed.")

        # Change only the trailing state letter (A → S)
        old_id = self.flexo_id
        self.flexo_id = self.flexo_id.with_state(EntityState.APPROVED_AND_SIGNED.value)
        self.origin = old_id
        return self

    def publish(self):
        """
        I move from APPROVED or APPROVED_AND_SIGNED to PUBLISHED.

        Uses allowed_transitions() to verify legality,
        then performs a version bump and lineage update.
        """
        if EntityState.PUBLISHED not in self.allowed_transitions():
            raise ValueError(
                f"Illegal state transition: {self.state.name} → PUBLISHED. "
            )

        new_fid = FlexOID.safe_generate(
            domain_id=self.domain_id,
            entity_type=self.entity_type.value,
            state=EntityState.PUBLISHED.value,
            text=self.text_seed,
            version=self.version
        )

        self.origin = self.flexo_id
        self.flexo_id = new_fid
        return self

    def obsolete(self):
        """I mark myself as obsolete"""
        if EntityState.OBSOLETE not in self.allowed_transitions():
            raise ValueError(
                f"Illegal state transition: {self.state.name} -> OBSOLETE. "
            )
        if self.state != EntityState.OBSOLETE:
            new_fid = FlexOID.safe_generate(
                domain_id=self.domain_id,
                entity_type=self.entity_type.value,
                state=EntityState.OBSOLETE.value,
                text=self.text_seed,
                version=self.version
            )

            self.origin = self.flexo_id
            self.flexo_id = new_fid

        return self

    def clone_new_base(self):
        """Start new lineage when obsolete."""
        self.origin = str(self.flexo_id)
        self.flexo_id = FlexOID.clone_new_base(
            domain_id=self.domain_id,
            entity_type=self.entity_type.value,
            state=EntityState.DRAFT.value,
            text=self.text_seed,
        )
        return self

    # ───────────────────────────────────────────────────────────────
    # Integrity verification
    # ───────────────────────────────────────────────────────────────
    @staticmethod
    def verify_integrity(entity) -> bool:
        """
        Verify that an entity's FlexOID, state, and fingerprint are consistent.

        Returns False if:
          - flexo_id format is invalid
        """
        try:
            if not isinstance(entity.flexo_id, FlexOID):
                return False

            # --- Fingerprint validation ---
            if hasattr(entity, "fingerprint") and entity.fingerprint:
                expected_fp = entity._compute_fingerprint()
                if expected_fp != entity.fingerprint:
                    return False

            return True

        except Exception as e:
            print(e)
            return False

    def allowed_transitions(self) -> list[str]:
        """Return a list of possible next state names (for GUI use)."""
        return self.state.allowed_transitions()

    def apply_state_change(self, new_state):
        """
        I safely apply the requested state transition by invoking
        the corresponding FlexoEntity method if available.
        """

        if self.state == new_state:
            return
        if new_state == EntityState.APPROVED:
            self.approve()
        elif new_state == EntityState.APPROVED_AND_SIGNED:
            self.sign()
        elif new_state == EntityState.PUBLISHED:
            self.publish()
        elif new_state == EntityState.OBSOLETE:
            self.obsolete()
        else:
            raise RuntimeError(f"No handler for state transition to {new_state}")
