import tkinter as tk
from tkinter import ttk, messagebox, simpledialog

from flexograder.core_entities.exam import ExamLayout
from flexograder.gui.create_exam_dialog import ExamDialog
from flexograder.gui.page_name_dialog import PageNameDialog

class ExamLayoutDialog(tk.Toplevel):
    """Dialog for arranging Exam layout: pages on the right, question pool on the left."""

    def __init__(self, parent, exam):
        super().__init__(parent)
        self.parent = parent
        self.exam = exam

        # Ensure we have a layout object
        self.layout = getattr(exam, "layout", None)
        if self.layout is None:
            self.layout = ExamLayout()

        self.title(f"Exam Layout — {exam.title}")
        self.geometry("900x600")
        self.configure(padx=10, pady=10)

        self.create_widgets()

        self.refresh_all()

        # Make modal
        self.transient(parent)
        self.grab_set()
        self.wait_visibility()
        self.focus_set()

    # ─────────────────────────────────────────────────────────────
    # GUI construction (pure grid layout)
    # ─────────────────────────────────────────────────────────────
    def create_widgets(self):
        # 3 main vertical rows: top section, page section, footer
        self.rowconfigure(0, weight=1)
        self.rowconfigure(1, weight=0)
        self.rowconfigure(2, weight=0)
        self.columnconfigure(0, weight=1)

        # ─────────────────────────────
        # TOP SECTION (Questions | Mid | Pages)
        # ─────────────────────────────
        top = ttk.Frame(self)
        top.grid(row=0, column=0, sticky="nsew", pady=(0, 8))
        top.columnconfigure(0, weight=1)   # left
        top.columnconfigure(1, weight=0)   # mid
        top.columnconfigure(2, weight=1)   # right
        top.rowconfigure(1, weight=1)

        # LEFT — Question pool
        ttk.Label(top, text="Exam Questions").grid(row=0, column=0, sticky="w")
        self.questions_list = tk.Listbox(top, selectmode="extended", exportselection=False)
        self.questions_list.grid(row=1, column=0, sticky="nsew", padx=(0, 6))

        # Filter — show only unassigned questions
        self.var_unassigned = tk.BooleanVar(value=False)
        self.chk_unassigned = ttk.Checkbutton(
            top,
            text="Show unassigned only",
            variable=self.var_unassigned,
            command=self.refresh_questions_list
        )
        self.chk_unassigned.grid(row=2, column=0, sticky="w", padx=(0, 6), pady=(4, 0))

        qbtns = ttk.Frame(top)
        qbtns.grid(row=3, column=0, sticky="ew", pady=(5, 0))
        ttk.Button(qbtns, text="Add Question", command=self.add_question).pack(side="left")
        ttk.Button(qbtns, text="Delete", command=self.delete_question).pack(side="left")

        # MIDDLE — Assignment controls
        mid = ttk.Frame(top)
        mid.grid(row=1, column=1, sticky="ns", padx=4)
        ttk.Button(mid, text="→ Add to Page", command=self.add_selected_to_page).pack(pady=10)
        ttk.Button(mid, text="← Remove", command=self.remove_selected_from_page).pack(pady=10)

        # RIGHT — Pages
        ttk.Label(top, text="Exam Pages").grid(row=0, column=2, sticky="w")
        self.pages_list = tk.Listbox(top, exportselection=False)
        self.pages_list.grid(row=1, column=2, sticky="nsew", padx=(6, 0))
        self.pages_list.bind("<<ListboxSelect>>", self.on_page_selected)

        pagebtns = ttk.Frame(top)
        pagebtns.grid(row=2, column=2, sticky="ew", pady=(5, 0))
        ttk.Button(pagebtns, text="Add Page", command=self.add_page).pack(side="left")
        ttk.Button(pagebtns, text="Rename", command=self.rename_page).pack(side="left")
        ttk.Button(pagebtns, text="Delete", command=self.delete_page).pack(side="left")
        ttk.Button(pagebtns, text="↑", command=self.move_page_up).pack(side="right")
        ttk.Button(pagebtns, text="↓", command=self.move_page_down).pack(side="right")

        # ─────────────────────────────
        # SECOND SECTION — Page contents
        # ─────────────────────────────
        page_frame = ttk.LabelFrame(self, text="Elements on selected page")
        page_frame.grid(row=1, column=0, sticky="nsew")
        page_frame.columnconfigure(0, weight=1)
        page_frame.rowconfigure(0, weight=1)

        self.page_questions = tk.Listbox(page_frame, height=8, exportselection=False)
        self.page_questions.grid(row=0, column=0, sticky="nsew", padx=2, pady=2)

        pqbtns = ttk.Frame(page_frame)
        pqbtns.grid(row=1, column=0, sticky="ew", pady=(4, 0))
        ttk.Button(pqbtns, text="Move Up", command=self.move_q_up).pack(side="left")
        ttk.Button(pqbtns, text="Move Down", command=self.move_q_down).pack(side="left")

        # ─────────────────────────────
        # FOOTER — OK / Cancel
        # ─────────────────────────────
        footer = ttk.Frame(self)

        footer.grid(row=2, column=0, sticky="ew", pady=(10, 0))
        ttk.Button(footer, text="Edit Exam Metadata…",
                   command=self.edit_metadata).pack(side="right", padx=5)
        ttk.Button(footer, text="OK", command=self.on_ok).pack(side="right", padx=5)
        ttk.Button(footer, text="Cancel", command=self.destroy).pack(side="right", padx=5)

    def edit_metadata(self):
        dialog = ExamDialog(parent=self, title="Edit Metadata", exam=self.exam)
        if dialog.result is None:
            return
        self.exam.title=dialog.result["title"]
        self.exam.author=dialog.result["author"]
        self.exam.intro_note=dialog.result["intro_note"]
        self.exam.submission_note=dialog.result["submission_note"]
        self.exam.allowed_aids=dialog.result["allowed_aids"]
        self.exam.duration=dialog.result["duration"]
        print(dialog.result["include_id_form"])
        if dialog.result["include_id_form"]:
            self.exam.add_default_id_form()
        else:
            messagebox.showinfo("Deletion impossible", "Once added IDForm can not be removed.")
        self.exam._update_fingerprint()

    # ─────────────────────────────────────────────────────────────
    # Refresh
    # ─────────────────────────────────────────────────────────────
    def refresh_all(self):

        self.refresh_pages_list()
        self.refresh_questions_list()

    def refresh_page_questions(self, page):
        self.page_questions.delete(0, tk.END)
        for qid in page.element_ids:
            q = self.exam.get_element_by_id(qid)
            if q:
                self.page_questions.insert(tk.END, q.text[:50])

    def refresh_pages_list(self):
        """Refresh the right-side pages list."""
        self.pages_list.delete(0, tk.END)
        for p in self.layout.pages:
            self.pages_list.insert(tk.END, str(p.title))

    def refresh_questions_list(self):
        """Refresh left-side question pool, considering the 'unassigned only' filter."""
        self.questions_list.delete(0, tk.END)

        # Collect all question IDs currently placed on pages
        assigned = set()
        for page in self.layout.pages:
            for qid in page.element_ids:
                assigned.add(qid)

        for q in self.exam.questions:
            if self.var_unassigned.get() and q.flexo_id in assigned:
                continue
            self.questions_list.insert(tk.END, q.text[:50])
    # ─────────────────────────────────────────────────────────────
    # Page operations
    # ─────────────────────────────────────────────────────────────
    def add_page(self):
        suggestions = self.exam.all_topics()
        dlg = PageNameDialog(self, suggestions)
        if not dlg.result:
            return
        self.layout.add_page(dlg.result)
        self.refresh_all()

    def rename_page(self):
        idx = self.pages_list.curselection()
        if not idx:
            return
        page = self.layout.pages[idx[0]]
        new_name = simpledialog.askstring("Rename Page", "New title:",
                                          initialvalue=page.title, parent=self)
        if new_name:
            page.title = new_name
            self.refresh_all()
            self.pages_list.selection_set(idx[0])

    def delete_page(self):
        idx = self.pages_list.curselection()
        if not idx:
            return
        del self.layout.pages[idx[0]]
        self.refresh_all()

    def move_page_up(self):
        idx = self.pages_list.curselection()
        if not idx or idx[0] == 0:
            return
        i = idx[0]
        pages = self.layout.pages
        pages[i - 1], pages[i] = pages[i], pages[i - 1]
        self.refresh_all()
        self.pages_list.selection_set(i - 1)

    def move_page_down(self):
        idx = self.pages_list.curselection()
        if not idx or idx[0] == len(self.layout.pages) - 1:
            return
        i = idx[0]
        pages = self.layout.pages
        pages[i + 1], pages[i] = pages[i], pages[i + 1]
        self.refresh_all()
        self.pages_list.selection_set(i + 1)

    # ─────────────────────────────────────────────────────────────
    # Question assignment
    # ─────────────────────────────────────────────────────────────
    def add_selected_to_page(self):
        idx_page = self.pages_list.curselection()
        if not idx_page:
            messagebox.showinfo("Select Page", "Please select a page first.")
            return
        page = self.layout.pages[idx_page[0]]
        for i in self.questions_list.curselection():
            q = self.exam.questions[i]
            if q.flexo_id not in page.element_ids:
                page.element_ids.append(q.flexo_id)
        self.refresh_page_questions(page)

    def remove_selected_from_page(self):
        idx_page = self.pages_list.curselection()
        if not idx_page:
            return
        page = self.layout.pages[idx_page[0]]
        for i in reversed(self.page_questions.curselection()):
            del page.element_ids[i]
        self.refresh_page_questions(page)

    def on_page_selected(self, event=None):
        idx = self.pages_list.curselection()
        if not idx:
            self.page_questions.delete(0, tk.END)
            return
        page = self.layout.pages[idx[0]]
        self.refresh_page_questions(page)

    # ─────────────────────────────────────────────────────────────
    # Question management
    # ─────────────────────────────────────────────────────────────
    def add_question(self):
        messagebox.showinfo("Info", "Use the main builder to add questions to the exam.")

    def delete_question(self):
        idxs = list(self.questions_list.curselection())
        if not idxs:
            return
        if not messagebox.askyesno("Confirm", "Remove selected questions from exam?"):
            return
        qids = [self.exam.questions[i].flexo_id for i in idxs]
        for qid in qids:
            self.exam.remove_element(qid)
            for p in self.layout.pages:
                if qid in p.element_ids:
                    p.element_ids.remove(qid)
        self.refresh_all()

    # ─────────────────────────────────────────────────────────────
    # Move question within page
    # ─────────────────────────────────────────────────────────────
    def move_q_up(self):
        """Move selected question one position up within the current page."""
        idx = self.page_questions.curselection()
        if not idx:
            return
        i = idx[0]
        page_idx = self.pages_list.curselection()
        if not page_idx:
            return
        page = self.layout.pages[page_idx[0]]
        if i == 0:
            return
        page.element_ids[i - 1], page.element_ids[i] = page.element_ids[i], page.element_ids[i - 1]
        self.refresh_page_questions(page)
        self.page_questions.selection_clear(0, tk.END)
        self.page_questions.selection_set(i - 1)

    def move_q_down(self):
        """Move selected question one position down within the current page."""
        idx = self.page_questions.curselection()
        if not idx:
            return
        i = idx[0]
        page_idx = self.pages_list.curselection()
        if not page_idx:
            return
        page = self.layout.pages[page_idx[0]]
        if i == len(page.element_ids) - 1:
            return
        page.element_ids[i + 1], page.element_ids[i] = page.element_ids[i], page.element_ids[i + 1]
        self.refresh_page_questions(page)
        self.page_questions.selection_clear(0, tk.END)
        self.page_questions.selection_set(i + 1)

    # ─────────────────────────────────────────────────────────────
    # Finalize
    # ─────────────────────────────────────────────────────────────
    def on_ok(self):
        self.exam.layout = self.layout
        self.destroy()
