from dataclasses import dataclass, field
from typing import Optional, List, Dict, Any
from xml.dom import minidom
from pathlib import Path
import zipfile
import html

from flexoentity import FlexoEntity, EntityType, EntityState, Domain
from .exam_elements import ExamElement, ChoiceQuestion, IDForm, element_factory
from .question_catalog import QuestionCatalog


def _esc(value) -> str:
    if value is None:
        return ""
    return html.escape(str(value), quote=True)


@dataclass
class ExamLayout:
    pages: list["ExamPage"] = field(default_factory=list)

    def to_dict(self) -> dict:
        """Serialize the full layout (list of pages + element IDs)."""
        return {
            "pages": [p.to_dict() for p in self.pages]
        }

    @classmethod
    def from_dict(cls, data: dict) -> "ExamLayout":
        """Rebuild layout from dictionary representation."""
        pages = [ExamPage.from_dict(p) for p in data.get("pages", [])]
        return cls(pages=pages)

    def add_page(self, title: str) -> "ExamPage":
        page = ExamPage(title=title)
        self.pages.append(page)
        return page

    def assign_element(self, page_title: str, qid: str):
        """Place an element (by ID) on a page."""
        page = next((p for p in self.pages if p.title == page_title), None)
        if not page:
            page = self.add_page(page_title)
        page.element_ids.append(qid)

    def find_page(self, title):
        return next((p for p in self.pages if p.title == title), None)

    def purge_element(self, qid):
        for p in self.pages:
            if qid in p.element_ids:
                p.element_ids.remove(qid)

@dataclass
class ExamPage:
    title: str
    element_ids: List[str] = field(default_factory=list)

    def to_dict(self):
        return {"title": self.title, "element_ids": [str(qid) for qid in self.element_ids]}

    @classmethod
    def from_dict(cls, data: dict):
        return cls(
            title=data.get("title", ""),
            element_ids=data.get("element_ids", []),
        )

    def to_html(self, *, exam, index: int, active: bool) -> str:
        cls = "page active" if active else "page"

        prev_btn = (
            "" if index == 0
            else '<button type="button" class="prevBtn">Zurück</button>'
        )

        body = []

        for eid in self.element_ids:
            el = exam.get_element_by_id(eid)

            if el is None:
                body.append(
                    f'<p class="missing-question">'
                    f'Element mit ID {_esc(eid)} nicht gefunden.</p>'
                )
            else:
                body.append(exam._render_element(el))

        return f"""
        <div class="{cls}" data-index="{index}">
        <h2>{_esc(self.title)}</h2>
        {''.join(body)}
        <div class="nav-buttons">
        {prev_btn}
        <button type="button" class="nextBtn">Weiter</button>
        </div>
        </div>
        """

# --- Exam Container ---
@dataclass
class Exam(FlexoEntity):

    ENTITY_TYPE = EntityType.CATALOG

    title: str = ""
    duration: str = ""
    allowed_aids: str = ""
    headline: str = ""
    intro_note: str = ""
    submit_note: str = ""
    author: str = "unknown"
    domains: Dict[str, Dict[str, Any]] = field(default_factory=dict)
    elements: dict[str, ExamElement] = field(default_factory=dict)
    layout: ExamLayout = field(default_factory=ExamLayout)

    @classmethod
    def default(cls):
        return cls()

    def add_element(self, element: ExamElement):
        self.elements[element.flexo_id] = element

    def remove_element(self, qid: str):
        self.elements.pop(qid, None)
        self.layout.purge_element(qid)

    def get_element_by_id(self, qid: str) -> Optional[ExamElement]:
        return self.elements.get(qid)

    def ensure_page(self, title: str) -> ExamPage:
        """
        Return an existing page by title, or create it if missing.
        Useful for layout editing or import operations.
        """
        page = self.layout.find_page(title)
        if not page:
            page = self.layout.add_page(title)
        return page

    @property
    def text_seed(self) -> str:
        q_ids = " ".join(self.elements.keys())
        return f"{self.title.strip()} {q_ids}".strip()

    def _deserialize_content(self, content: dict):

        self.title = content.get("title", "")
        self.headline = content.get("headline", "")
        self.intro_note = content.get("intro_note", "")
        self.submit_note = content.get("submit_note", "")
        self.allowed_aids = content.get("allowed_aids", "")
        self.duration = content.get("duration", "")
        self.author = content.get("author", "")
        self.domains = [Domain.from_dict(d) for d in content.get("domains", [])]
        self.elements = {
            e.flexo_id: e
            for e in (element_factory(q) for q in content.get("elements", []))
        }
        layout_data = content.get("layout", {}) or {}
        pages = [ExamPage.from_dict(p) for p in layout_data.get("pages", [])]
        self.layout = ExamLayout(pages=pages)

    def _serialize_content(self):
        """
        Return the exam’s structural content:
        - basic exam attributes
        - page layout
        - elements references (not full objects)
        """

        return {
            "title": self.title,
            "headline": self.headline,
            "intro_note": self.intro_note,
            "submit_note": self.submit_note,
            "allowed_aids": self.allowed_aids,
            "duration": self.duration,
            "author": self.author,
            "domains": [d.to_dict() for d in self.domains],
            "elements": [e.to_dict() for e in self.elements.values()],
            "layout": self.layout.to_dict(),
        }

    # --- Core API ---
    def add_page(self, page_title):
        self.layout.add_page(page_title)

    def add_element_to_page(self, page_title: str, element: ExamElement):
        """Assign an element to a page."""
        self.layout.assign_element(page_title, element.flexo_id)

    def shuffle_options(self):
        for each_question in self.elements.values():
            if isinstance(each_question, ChoiceQuestion):
                each_question.shuffle_answers()

    def has_id_form(self) -> bool:
        """Return True if the exam contains an IDForm element."""
        return any(isinstance(e, IDForm) for e in self.elements.values())

    @property
    def is_tainted(self) -> bool:

        return any(e.state != EntityState.APPROVED for e in self.elements.values())

    def referenced_media_paths(self) -> set[str]:
        refs = set()
        for q in self.elements.values():
            if q.has_media():
                for m in q.get_media():
                    if m:
                        refs.add(m.src)
        return refs

    @property
    def questions(self) -> list[ExamElement]:
        return [
            e for e in self.elements.values()
            if isinstance(e, ChoiceQuestion)
        ]

    def all_topics(self):
        topics = [each.topic for each in self.questions]
        topics.extend([each.subtopic for each in self.questions])
        return list(set(topics))
 
    def to_question_catalog(self):
        return QuestionCatalog.with_domain_id(domain_id=self.domain_id,
                                              title=self.title, author=self.author,
                                              domains=self.domains,
                                              questions=self.questions,
                                              )

    def _render_element(self, el) -> str:

        def _esc(value) -> str:
            if value is None:
                return ""
            return html.escape(str(value), quote=True)

        return (
            f'<div class="exam-element" '
            f'data-qid="{_esc(el.flexo_id)}" '
            f'data-qtype="{_esc(el.subtype)}">'
            f'{el.to_html()}'
            f'</div>'
        )

    def _render_header(self) -> str:
        return f"""
        <h1>
        {_esc(self.title)} – <span id="studentNameDisplay">(unbekannt)</span>
        </h1>
        <h2 class="exam-headline">{_esc(self.headline)}</h2>
        <p><strong>Dauer:</strong> {_esc(self.duration)}</p>
        <p><strong>Erlaubte Hilfsmittel:</strong> {_esc(self.allowed_aids)}</p>
        <p>{_esc(self.intro_note)}</p>
        """

    def _render_navigation(self) -> str:
        items = []

        for idx, page in enumerate(self.layout.pages):
            items.append(
                f'<li><a href="#" class="nav-link" data-index="{idx}">'
                f'{idx + 1}. {_esc(page.title)}</a></li>'
            )

        # final page
        items.append(
            f'<li><a href="#" class="nav-link" data-index="{len(self.layout.pages)}">'
            f'Speichern</a></li>'
        )

        return f"""
        <nav class="page-nav">
        <h2>Seiten</h2>
        <ul>
        {''.join(items)}
        </ul>
        </nav>
        """

    def _render_final_page(self) -> str:
        idx = len(self.layout.pages)
        return f"""
        <div class="page" data-index="{idx}">
        <h3>Fertig zum Absenden?</h3>
        <p class="final-warning">{_esc(self.submit_note)}</p>
        <div class="nav-buttons">
        <button type="button" class="prevBtn">Zurück zur Bearbeitung</button>
        <button type="submit" id="finalizeBtn">Jetzt speichern</button>
        </div>
        </div>
        """

    def _render_form(self) -> str:
        pages = []

        for idx, page in enumerate(self.layout.pages):
            pages.append(
                page.to_html(
                    exam=self,
                    index=idx,
                    active=(idx == 0),
                )
            )

        pages.append(self._render_final_page())

        return f"""
        <form id="examForm">
        {''.join(pages)}
        </form>
        """

    # FIXME: This introduces bugs due to silently removing spaces
    def pretty_html(self, html_str: str, indent: str = "  ") -> str:
        """
        Pretty-print HTML using minidom.

        WARNING:
        - Requires XHTML-compatible HTML
        - Intended for debugging / inspection only
        """
        # minidom requires a single root node
        if not html_str.lstrip().startswith("<"):
            raise ValueError("Not HTML/XML")

        dom = minidom.parseString(html_str)
        pretty = dom.toprettyxml(indent=indent)

        # Remove empty lines introduced by minidom
        return "\n".join(
            line for line in pretty.splitlines()
            if line.strip()
        )

    def to_html(self) -> str:
        return f"""
        <!DOCTYPE html
        PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN"
        "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
        <html xmlns="http://www.w3.org/1999/xhtml" lang="de">
        <head>
        <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
        <title>{_esc(self.title)}</title>
        <link rel="stylesheet" href="static/exam.css" />
        <script src="static/exam.js" defer="defer"></script>
        </head>
        <body data-exam-flexo-id="{_esc(self.flexo_id)}">
        {self._render_header()}
        {self._render_navigation()}
        {self._render_form()}
        </body>
        </html>
        """

    def to_zip(self, html_str: str, media_root, css, output_dir):
        media_root = Path(media_root).resolve()
        css = Path(css)
        script = css.parent / "exam.js"
        output_dir = Path(output_dir)

        # make a safe zip filename
        zip_name = f"{self.title}_{self.flexo_id}".replace("/", "-").replace("\\", "-")
        archive_name = output_dir / f"{zip_name}.zip"

        with zipfile.ZipFile(archive_name, "w", zipfile.ZIP_DEFLATED) as zipf:
            # HTML inside the zip
            zipf.writestr("exam.html", html_str)

            # Only referenced media, relative to media_root, stored under 'media/...'
            for rel in self.referenced_media_paths():
                # normalize to a relative Path
                # strip any accidental leading slashes or 'media/' prefix
                rel = Path(str(rel).lstrip("/\\"))
                if str(rel).startswith("media/"):
                    rel = Path(str(rel)[6:])  # backward-compat if older JSON kept 'media/...'

                abs_path = (media_root / rel).resolve()

                # prevent path traversal: abs_path must stay within media_root
                if media_root not in abs_path.parents and abs_path != media_root:
                    print(f"Skipping suspicious media path outside root: {abs_path}")
                    continue

                if abs_path.exists():
                    arcname = Path("media") / rel  # path inside the ZIP (relative)
                    # use POSIX path separators inside zip
                    zipf.write(abs_path, arcname=str(arcname.as_posix()))
                else:
                    print(f"Missing media file: {abs_path}")

            # CSS inside zip at a fixed relative location
            if css.exists():
                zipf.write(css, arcname="static/exam.css")
            else:
                print(f"Missing CSS: {css}")

            print("Script:", script)
            if script.exists():
                zipf.write(script, arcname="static/exam.js")
            else:
                raise FileNotFoundError
        return archive_name

    def add_default_id_form(self, page_title: str = "Candidate Information") -> "IDForm":
        """
        Ensure a default IDForm element exists in the exam and is placed in the layout.
        If one already exists, return it unchanged.
        """

        # 1. Check if an IDForm already exists
        for e in self.elements.values():
            if isinstance(e, IDForm):
                return e

        # 2. Create a new IDForm
        id_form = IDForm.with_domain_id(
            domain_id=self.domain_id,
            text="Please enter your name and candidate ID.",
            fields=[
                {
                    "id": "first_name",
                    "label": "Vorname",
                    "required": True
                },
                {
                    "id": "last_name",
                    "label": "Nachname",
                    "required": True
                },
                {
                    "id": "personal_id",
                    "label": "Candidate ID",
                    "required": True
                }
            ],
        )

        # 3. Add to element registry
        self.add_element(id_form)

        # 4. Ensure target page exists
        page = self.ensure_page(page_title)

        # 5. Place element on page (once)
        if id_form.flexo_id not in page.element_ids:
            page.element_ids.insert(0, id_form.flexo_id)

        return id_form

    def evaluate(self, submission):
        """
        Evaluate answers using the current layout order.
        Falls back gracefully if a page references a missing question.
        """
        total_score = 0.0
        max_score = 0.0
        detailed = []

        for page in self.layout.pages:
            for qid in page.element_ids:
                print("QID:", qid)
                q = self.get_element_by_id(qid)
                print("Question:", q)
                if not q or not isinstance(q, ChoiceQuestion):
                    print("Skip")
                    continue
                # prefer flexo_id as the answer key

                print("FlexoID:", q.flexo_id)
                submitted = submission.get_answer_for(q.flexo_id)
                print("Submitted:", submitted)
                score = q.evaluate(submitted)
                total_score += score
                max_score += getattr(q, "max_score", 1.0)
                detailed.append({
                    "element_id": qid,
                    "score": score,
                    "max_score": getattr(q, "max_score", 1.0),
                })

        return {
            "total_score": total_score,
            "max_score": max_score,
            "percentage": (total_score / max_score) * 100 if max_score else 0.0,
            "details": detailed,
        }
