# flexoentity/domain_manager.py

from __future__ import annotations

from typing import Iterable

from .flexo_collection import FlexoCollection
from .domain import Domain
from .flexo_entity import EntityState


class DomainNotFoundError(KeyError):
    pass


class InvalidDomainError(ValueError):
    pass


class DuplicateDomainError(ValueError):
    pass


class DomainManager:
    """
    Manager for Domain entities, implemented using composition.

    - Uses FlexoCollection internally.
    - Domains are keyed by domain.domain_id (explicit override).
    - No inheritance from FlexoCollection.
    - Responsible ONLY for domain-specific rules.
    """

    def __init__(self, registry):
        # store is a general-purpose collection
        self._store = FlexoCollection()
        self._registry = registry

    # ---------------------------------------------------------------
    # Core API
    # ---------------------------------------------------------------

    def add(self, domain: Domain) -> None:
        """
        Add a domain using domain.domain_id as key.
        """
        if not isinstance(domain, Domain):
            raise TypeError(
                f"DomainManager accepts only Domain instances, got {type(domain)}"
            )
        if domain.domain_id in self._store:
            raise DuplicateDomainError(f"Domain '{domain.domain_id} already exists.")
        self._store.add(domain, key=domain.domain_id)

    def get(self, domain_id: str) -> Domain:
        """Retrieve a domain or raise DomainNotFoundError."""
        if isinstance(domain_id, Domain):
            raise TypeError("DomainManager.get() expects a domain_id string, not a Domain object.")
        result = self._store.get(domain_id)
        if result is None:
            raise DomainNotFoundError(f"Domain '{domain_id}' not found.")
        return result

    def can_delete(self, domain_id):
        return not any(self._registry.entities_by_domain(domain_id))

    def delete(self, domain_id):
        if not self.can_delete(domain_id):
            raise ValueError(f"Domain {domain_id} is still referenced.")
        self.remove(domain_id)

    def remove(self, domain_id: str) -> None:
        """Remove a domain (rarely used, mostly for tests)."""
        if domain_id not in self._store:
            raise DomainNotFoundError(domain_id)
        self._store.remove(domain_id)

    def all(self) -> list[Domain]:
        """List of all domains."""
        return self._store.entities()

    def all_domain_ids(self):
        return self._store.ids() 

    def clear(self):
        self._store.clear()

    # ---------------------------------------------------------------
    # Lifecycle / Approval Helpers
    # ---------------------------------------------------------------

    def ensure_approved(self, domain_id: str) -> None:
        """
        Used during entity.transition_to().
        Raises if the domain does not exist or is not APPROVED.
        """
        dom = self.get(domain_id)

        if dom.state != EntityState.APPROVED:
            raise InvalidDomainError(
                f"Domain '{domain_id}' must be APPROVED before "
                f"entities in this domain may be promoted. "
                f"(Current state: {dom.state})"
            )

    # ---------------------------------------------------------------
    # Representations
    # ---------------------------------------------------------------

    def __len__(self) -> int:
        return len(self._store)

    def __contains__(self, domain_id: str) -> bool:
        return domain_id in self._store

    def __repr__(self) -> str:
        return f"<DomainManager domains={self._store.keys()}>"
