# tests/stubs/single_choice_question.py
import pytest
from datetime import datetime
from dataclasses import dataclass, field
from typing import List
from flexoentity import FlexoEntity, EntityType, EntityState, Domain

@pytest.fixture
def fixed_datetime(monkeypatch):
    class FixedDate(datetime):
        @classmethod
        def now(cls, tz=None):
            return datetime(2025, 11, 1, tzinfo=tz)
    monkeypatch.setattr("flexoentity.id_factory.datetime", FixedDate)
    return FixedDate


@dataclass
class AnswerOption:
    id: str
    text: str
    points: float = 0.0

    def to_dict(self):
        return {"id": self.id, "text": self.text, "points": self.points}

    @classmethod
    def from_dict(cls, data):
        return cls(
            id=data.get("id", ""),
            text=data.get("text", ""),
            points=data.get("points", 0.0)
        )


@dataclass
class SingleChoiceQuestion(FlexoEntity):
    """A minimal stub to test FlexoEntity integration."""
    text: str = ""
    options: List[AnswerOption] = field(default_factory=list)


    @classmethod
    def default(cls):
        return cls(domain=Domain(domain="GEN",
                                 entity_type=EntityType.DOMAIN,
                                 state=EntityState.DRAFT),
                   state=EntityState.DRAFT, entity_type=EntityType.ITEM)

    def to_dict(self):
        base = super().to_dict()
        base.update({
            "text": self.text,
            "options": [opt.to_dict() for opt in self.options],
        })
        return base

    @property
    def text_seed(self) -> str:
        """Include answer options (and points) for deterministic ID generation."""

        joined = "|".join(
            f"{opt.text.strip()}:{opt.points}"
            for opt in sorted(self.options, key=lambda o: o.text.strip().lower())
        )
        return f"{self.text}{joined}"

    @classmethod
    def from_dict(cls, data):
        obj = cls(
            text=data.get("text", ""),
            options=[AnswerOption.from_dict(o) for o in data.get("options", [])],
        )
        # restore FlexoEntity core fields
        obj.domain = data.get("domain")
        obj.entity_type = EntityType[data.get("etype")] if "etype" in data else EntityType.ITEM
        obj.state = EntityState[data.get("state")] if "state" in data else EntityState.DRAFT
        if "flexo_id" in data:
            from flexoentity import FlexOID
            obj.flexo_id = FlexOID.parsed(data["flexo_id"])
        return obj

@pytest.fixture
def domain():
    return Domain.default()

@pytest.fixture
def sample_question():
    return SingleChoiceQuestion(domain=Domain.default(),
                               text="What is 2 + 2?",
                               options=[],
                               entity_type=EntityType.ITEM,
                               state=EntityState.DRAFT)
