"""
Persistence and integrity verification tests for Flex-O entities.
Ensures fingerprints survive JSON export/import and detect tampering.
"""
import json
import pytest

from flexoentity import FlexOID, EntityType, EntityState
from tests.conftest import DummyEntity


# ──────────────────────────────────────────────────────────────────────────────
@pytest.fixture
def approved_entity():
    """A fully published dummy entity for persistence tests."""
    e = DummyEntity(
        domain="AF",
        etype=EntityType.QUESTION,
        state=EntityState.DRAFT,
        seed="What is Ohm’s law?"
    )
    e.approve()
    e.sign()
    e.publish()
    return e

@pytest.mark.skip(reason="FlexOIDs are regenerated on import; enable once JSON format is stable")
def test_json_roundtrip_preserves_integrity(approved_entity):
    """
    Export to JSON and reload — ensure fingerprints remain valid.
    """
    json_str = approved_entity.to_json()
    loaded = approved_entity.__class__.from_json(json_str)

    # Fingerprint and state should match — integrity must pass
    assert approved_entity.__class__.verify_integrity(loaded)

    # Metadata should be preserved exactly
    assert approved_entity.flexo_id.signature == loaded.flexo_id.signature
    assert approved_entity.flexo_id == loaded.flexo_id
    assert loaded.state == approved_entity.state

# ──────────────────────────────────────────────────────────────────────────────

@pytest.mark.skip(reason="FlexOIDs regenerated on import; tampering detection not applicable yet")
def test_json_tampering_detection(approved_entity):
    """Tampering with content should invalidate fingerprint verification."""
    json_str = approved_entity.to_json()
    tampered_data = json.loads(json_str)
    tampered_data["text_seed"] = "Tampered content injection"
    tampered_json = json.dumps(tampered_data)

    # We use DummyEntity.from_json to reconstruct (FlexoEntity is abstract)
    loaded = approved_entity.__class__.from_json(tampered_json)
    assert not approved_entity.__class__.verify_integrity(loaded)


# ──────────────────────────────────────────────────────────────────────────────

@pytest.mark.skip(reason="FlexOIDs regenerated on import; corruption detection not yet applicable")
def test_json_file_corruption(approved_entity, tmp_path):
    """Simulate file corruption — integrity check must fail."""
    file = tmp_path / "entity.json"
    json_str = approved_entity.to_json()
    file.write_text(json_str)

    # Corrupt the file
    corrupted = json_str.replace("Ohm’s", "Omm’s")
    file.write_text(corrupted)

    loaded = approved_entity.__class__.from_json(file.read_text())
    assert not approved_entity.__class__.verify_integrity(loaded)
