import json
from .persistance_backend import PersistanceBackend


class SQLiteEntityBackend(PersistanceBackend):
    """
    SQLite backend storing **dicts**, not entities.
    Managers do the entity conversions.
    """

    def __init__(self, entity_class, conn, table_name):
        self.entity_class = entity_class
        self.conn = conn
        self.table = table_name
        self._init_schema()

    def _init_schema(self):
        self.conn.execute(f"""
            CREATE TABLE IF NOT EXISTS {self.table} (
                flexo_id TEXT PRIMARY KEY,
                json     TEXT NOT NULL
            )
        """)
        self.conn.commit()

    def save(self, entity):
        entity_dict = entity.to_dict()
        fid = entity_dict["meta"]["flexo_id"]
        self.conn.execute(
            f"INSERT OR REPLACE INTO {self.table} (flexo_id, json) VALUES (?, ?)",
            (fid, json.dumps(entity_dict))
        )
        self.conn.commit()

    def update(self, entity):
        entity_dict = entity.to_dict()
        fid = entity_dict["meta"]["flexo_id"]
        self.conn.execute(
            f"UPDATE {self.table} SET json = ? WHERE flexo_id = ?",
            (json.dumps(entity_dict), fid)
        )
        self.conn.commit()

    def delete(self, flexo_id):
        self.conn.execute(
            f"DELETE FROM {self.table} WHERE flexo_id = ?",
            (flexo_id,)
        )
        self.conn.commit()

    def load(self, flexo_id):
        row = self.conn.execute(
            f"SELECT json FROM {self.table} WHERE flexo_id = ?",
            (flexo_id,),
        ).fetchone()

        if not row:
            return None

        return self.entity_class.from_dict(json.loads(row["json"]))

    def load_all(self):
        rows = self.conn.execute(
            f"SELECT json FROM {self.table}"
        ).fetchall()

        return [self.entity_class.from_dict(json.loads(r["json"])) for r in rows]

    def clear(self):
        self.conn.execute(f"DELETE FROM {self.table}")
        self.conn.commit()
