from dataclasses import dataclass
from flexoentity.flexo_entity import FlexOID, FlexoEntity, EntityType, EntityState


@dataclass
class Domain(FlexoEntity):
    fullname: str = ""
    description: str = ""
    classification: str = "UNCLASSIFIED"
    owner: str = "unknown"

    @classmethod
    def default(cls):
        return cls(domain="GEN")

    def __post_init__(self):
        super().__post_init__()

        # Generate FlexOID only if missing
        if not getattr(self, "flexo_id", None):
            domain_code = self.domain or "GEN"
            self.flexo_id = FlexOID.safe_generate(
                domain=domain_code,
                entity_type=EntityType.DOMAIN.value,   # 'D'
                estate=EntityState.DRAFT.value,      # 'D'
                text=self.text_seed,
                version=1,
            )


        
    @property
    def text_seed(self) -> str:
        """Deterministic text seed for ID generation."""
        return f"{self.fullname}|{self.classification}|{self.owner}"

    def to_dict(self):
        base = super().to_dict()
        base.update({
            "fullname": self.fullname,
            "description": self.description,
            "classification": self.classification,
            "owner": self.owner
        })
        return base

    @classmethod
    def from_dict(cls, data):
        return cls(
            fullname=data.get("fullname", ""),
            description=data.get("description", ""),
            classification=data.get("classification", "UNCLASSIFIED"),
            owner=data.get("owner", "unknown"),
        )
