from .flexo_entity import FlexoEntity
from .typed_collection import TypedCollection


class EntityNotFoundError(KeyError):
    pass


class InvalidEntityError(ValueError):
    pass


class DuplicateEntityError(ValueError):
    pass


class EntityManager:
    """
    Backend-agnostic manager for any FlexOEntity subclass.

    Responsibilities:
      - enforce ENTITY_CLASS (e.g., FlexoUser, Domain, Question)
      - convert entities <-> dicts
      - delegate persistence to backend
      - provide a consistent CRUD API for GUIs and CLI
      - avoid backend-specific code in subclasses
    """

    ENTITY_CLASS = None  # must be overridden by subclasses

    def __init__(self, local_backend, staging_backend, permanent_backend):
        if self.ENTITY_CLASS is None:
            raise ValueError("Subclasses must define ENTITY_CLASS")

        if not issubclass(self.ENTITY_CLASS, FlexoEntity):
            raise TypeError("ENTITY_CLASS must be a subclass of FlexOEntity")
        self.local_backend = local_backend
        self.staging_backend = staging_backend
        self.permanent_backend = permanent_backend

    def backend_of_domain_id(self, domain_id: str):
        for backend_name, backend in [
                ("local", self.local_backend),
                ("staging", self.staging_backend),
                ("permanent", self.permanent_backend),
        ]:
            for dom in backend.load_all():
                if dom.domain_id == domain_id:
                    return backend_name
        return None

    def backend_of_flexo_id(self, flexo_id: str):
        for backend_name, backend in [
                ("local", self.local_backend),
                ("staging", self.staging_backend),
                ("permanent", self.permanent_backend),
        ]:
            if backend.load(flexo_id) is not None:
                return backend_name
        return None

    # ------------------------------------------------------------------
    # CRUD operations
    # ------------------------------------------------------------------

    def add(self, entity):
        """Insert a new entity."""
        self._ensure_type(entity)
        self.local_backend.save(entity)

    def update(self, entity):
        """Replace an existing entity (same flexo_id)."""
        self._ensure_type(entity)
        self.local_backend.update(entity)

    def delete(self, flexo_id: str):
        """Remove entity by string flexo_id."""
        self.local_backend.delete(flexo_id)

    # ------------------------------------------------------------------
    # Retrieval
    # ------------------------------------------------------------------

    def get(self, flexo_id: str):
        """
        Load entity by flexo_id str.
        Returns ENTITY_CLASS instance or None.
        """
        return self.local_backend.load(flexo_id)

    # FIXME: Readd staging backend later
    
    def all(self):
        """Load all entities as a list of instances."""
        return (self.local_backend.load_all() + # self.staging_backend.load_all() +
                self.permanent_backend.load_all())

    def promote_to_staging(self, flexo_id):
        entity = self.local_backend.load(flexo_id)
        if entity is None:
            raise EntityNotFoundError
        self.staging_backend.save(entity)
        self.local_backend.delete(flexo_id)

    def promote_to_permanent(self, flexo_id):
        entity = self.staging_backend.load(flexo_id)
        if entity is None:
            raise EntityNotFoundError
        self.permanent_backend.save(entity)
        self.staging_backend.delete(flexo_id)

    def sync_all(self):
        # sync staging → local
        for e in self.staging_backend.load_all():
            self.local_backend.save(e)

        # sync permanent → local
        for e in self.permanent_backend.load_all():
            self.local_backend.save(e)
        self.refresh()

    # ------------------------------------------------------------------
    # Helpers
    # ------------------------------------------------------------------

    def exists(self, flexo_id: str) -> bool:
        return self.get(flexo_id) is not None

    def count(self) -> int:
        return len(self.local_backend.load_all())

    def add_or_update(self, entity):
        """
        Convenience for GUIs:
        Insert or overwrite based on whether flexo_id exists.
        """
        if self.exists(entity.flexo_id):
            self.update(entity)
        else:
            self.add(entity)

    def as_collection(self):
        return TypedCollection(self.ENTITY_CLASS, items=self.local_backend.load_all())

    # ------------------------------------------------------------------
    # Internal
    # ------------------------------------------------------------------

    def _ensure_type(self, entity):
        if not isinstance(entity, self.ENTITY_CLASS):
            raise TypeError(
                f"Expected {self.ENTITY_CLASS.__name__}, "
                f"got {type(entity).__name__}"
            )

    # ------------------------------------------------------------------

    def __repr__(self):
        return f"<FlexoEntityManager for {self.ENTITY_CLASS.__name__}>"
