# tests/conftest.py

import pytest
import json
from flexoentity import FlexoEntity, EntityType, EntityState, Domain


class DummyEntity(FlexoEntity):
    """Minimal concrete subclass for testing FlexoEntity logic."""

    def __init__(self, domain, etype, state, seed="DUMMY"):
        self._seed = seed
        super().__init__(domain, etype, state)

    @property
    def text_seed(self) -> str:
        return self._seed 

    @classmethod
    def from_dict(cls, data):
        """Ensure enums and seed are reconstructed correctly."""
        domain = data["domain"]
        etype = EntityType[data["etype"]] if isinstance(data["etype"], str) else data["etype"]
        state = EntityState[data["state"]] if isinstance(data["state"], str) else data["state"]
        seed = data.get("text_seed", "DUMMY-CONTENT")
        return cls(domain=domain, etype=etype, state=state, seed=seed)

    @classmethod
    def from_json(cls, data_str: str):
        return cls.from_dict(json.loads(data_str))
    
@pytest.fixture
def entity():
    """Generic FlexoEntity-like instance in draft state."""
    return DummyEntity(
        domain=Domain(domain="SIG", etype=EntityType.DOMAIN, state=EntityState.DRAFT, fullname="Signal Corps", classification="RESTRICTED"),
        etype=EntityType.CATALOG,
        state=EntityState.DRAFT,
    )

@pytest.fixture
def sample_exam():
    """Reusable Exam fixture with default metadata for tests."""
    exam = Exam(
        domain="AF",
        etype=EntityType.EXAM,
        state=EntityState.DRAFT,
        title="Signals Exam",
        duration="30 min",
        allowed_aids="none",
        headline="Basics of Signals",
        intro_note="Welcome to the exam",
        submit_note="Submit carefully",
        author="Tester",
    )
    return exam

@pytest.fixture
def null_media():
    """Provide a default NullMediaItem instance for tests."""
    return NullMediaItem(
        domain="GEN",
        etype=EntityType.MEDIA,
        state=EntityState.DRAFT
    )
