"""
Persistence and integrity verification tests for Flex-O entities.
Ensures fingerprints survive JSON export/import and detect tampering.
"""
import json
import pytest

from flexoentity import Domain


@pytest.fixture
def approved_domain(sample_domain):
    """Provide a fully approved and published Domain for persistence tests."""
    sample_domain.approve()
    sample_domain.sign()
    sample_domain.publish()
    return sample_domain


def test_json_roundtrip_preserves_integrity(approved_domain):
    """
    Export to JSON and reload — ensure fingerprints and signatures remain valid.
    """
    json_str = approved_domain.to_json()
    loaded = Domain.from_json(json_str)

    # Fingerprint and state should match — integrity must pass
    assert Domain.verify_integrity(loaded)

    # Metadata should be preserved exactly
    assert approved_domain.fingerprint == loaded.fingerprint
    assert approved_domain.flexo_id == loaded.flexo_id
    assert loaded.state == approved_domain.state


def test_json_tampering_detection(approved_domain):
    """Tampering with content should invalidate fingerprint verification."""
    json_str = approved_domain.to_json()
    tampered = json.loads(json_str)
    tampered["content"]["fullname"] = "Tampered content injection"
    tampered_json = json.dumps(tampered)

    loaded = Domain.from_json(tampered_json)
    assert not Domain.verify_integrity(loaded)


def test_json_file_corruption(approved_domain, tmp_path):
    """Simulate file corruption — integrity check must fail."""
    file = tmp_path / "question.json"
    json_str = approved_domain.to_json()
    print("JSON", json_str)
    file.write_text(json_str)

    # Corrupt the file (simulate accidental byte modification)
    corrupted = json_str.replace("ARITHMETIC", "ARITHM")
    file.write_text(corrupted)

    loaded = Domain.from_json(file.read_text())
    assert not Domain.verify_integrity(loaded)
