"""
id_factory.py — Central Flex-O ID generator and versioning control (hardened).

Improvements:
- BLAKE2s hashing (modern, fast, stdlib)
- 6 hex-digit hash (≈16.7M combinations)
- UTC-based dates for consistency
- Collision disambiguator (-A, -B, ...)
- Canonical seed and content fingerprint helpers
"""

import logging
from datetime import datetime, timezone
import hashlib
import secrets
import itertools
import json

logger = logging.getLogger(__name__)

# ──────────────────────────────────────────────────────────────────────────────
#  Canonicalization helpers
# ──────────────────────────────────────────────────────────────────────────────


def canonical_seed(obj) -> str:
    """
    Deterministically flatten an entity's core data into a string
    for hashing and deduplication.
    """
    if isinstance(obj, str):
        text = " ".join(obj.split())
        return text
    if isinstance(obj, dict):
        return json.dumps(obj, sort_keys=True, separators=(",", ":"))
    if hasattr(obj, "__dict__"):
        return canonical_seed(obj.__dict__)
    return str(obj)

# ──────────────────────────────────────────────────────────────────────────────
#  ID Factory
# ──────────────────────────────────────────────────────────────────────────────


class FlexOID:
    MAX_VERSION = 999
    WARN_THRESHOLD = 900

    # keep in-memory registry for same-session collisions (optional)
    _seen_hashes = set()

    @classmethod
    def from_string(cls, id_str: str):
        return cls(id_str)

    @classmethod
    def from_oid_and_version(cls, oid, version: int):
        if not (1 <= version <= cls.MAX_VERSION):
            raise ValueError(f"Version {version} out of bounds (1..{cls.MAX_VERSION}).")
        return FlexOID(f"{oid.prefix}@{version:03d}{oid.state_code}")

    def __init__(self, flexo_id: str):
        self.flexo_id = flexo_id

    def __eq__(self, other):
        if not isinstance(other, FlexOID):
            return NotImplemented
        return self.flexo_id == other.flexo_id

    def __lt__(self, other):
        if not isinstance(other, FlexOID):
            return NotImplemented
        if self.prefix != other.prefix:
            raise ValueError("Cannot order FlexOIDs from different prefixes")
        return self.version < other.version
        
    def __hash__(self):
        return hash(self.flexo_id)

    @staticmethod
    def _blake_hash(text: str) -> str:
        """Return a 12-hex BLAKE2s digest."""
        return hashlib.blake2s(text.encode("utf-8"), digest_size=6).hexdigest().upper()  # 6 bytes → 12 hex

    @staticmethod
    def safe_generate(domain, etype, estate, text, version=1, repo=None):
        """
        Generate a new FlexOID with deterministic hashing, handling rare collisions.
        """

        # Normalize domain (Domain object or string)
        domain_code = getattr(domain, "domain", domain)

        # Generate the deterministic candidate OID
        oid = FlexOID.generate(domain_code, etype, estate, text, version=version)

        # Collision detection — only if a repository is available
        if repo is not None:
            existing = repo.get(str(oid)) if hasattr(repo, "get") else repo.get(oid)
        else:
            existing = None

        if existing:
            try:
                same_seed = existing.text_seed == text or \
                            existing.canonical_seed() == canonical_seed(text)
            except Exception:
                same_seed = False

            if not same_seed:
                # Collision detected — regenerate deterministically
                print("Collision detected", len(repo))
                logger.warning(f"FlexOID collision detected for {oid}")

                # (A) refresh date
                date_part = datetime.now(timezone.utc).strftime("%y%m%d")

                # (B) add minimal deterministic salt (2 hex chars)
                salt = secrets.token_hex(1)
                salted_text = f"{text}|salt:{salt}"

                # (C) generate new OID with new date and salted seed
                oid = FlexOID.generate(
                    domain_code,
                    etype,
                    estate,
                    salted_text,
                    version=version,
                )

                # (D) record lineage if the caller has `origin` tracking
                if hasattr(existing, "flexo_id"):
                    logger.info(f"New lineage created from {existing.flexo_id}")

        return oid
    
    @staticmethod
    def generate(domain: str, etype: str, estate: str, text: str,
             version: int = 1):
        """
        Generate a deterministic Flex-O ID.

        - The hash (and therefore prefix) depends only on domain, etype, and text.
        → Prefix stays stable across state changes.
        """

        if not (1 <= version <= FlexOID.MAX_VERSION):
            raise ValueError(f"Version {version} exceeds limit; mark obsolete.")

        date_part = datetime.now(timezone.utc).strftime("%y%m%d")

        # state-independent hash seed → prefix stability
        hash_seed = canonical_seed(f"{domain}:{etype}:{text}")
        base_hash = FlexOID._blake_hash(hash_seed)

        ver_part = f"{version:03d}{estate}"
        flexo_id_str = f"{domain}-{etype}{date_part}-{base_hash}@{ver_part}"

        return FlexOID(flexo_id_str)

    # ──────────────────────────────────────────────────────────────────────────

    @property
    def state_code(self):
        part = self.flexo_id.rsplit("@", 1)[-1]

        if not (part and part[-1].isalpha()):
            raise ValueError(f"Invalid Flex-O ID format: {self.flexo_id}")
        return part[-1]

    # ──────────────────────────────────────────────────────────────────────────
    #  Parsed Accessors
    # ──────────────────────────────────────────────────────────────────────────

    @property
    def domain(self) -> str:
        """Return the domain prefix (e.g., 'AF')."""
        try:
            return self.flexo_id.split('-', 1)[0]
        except IndexError:
            raise ValueError(f"Malformed Flex-O ID: {self.flexo_id}")

    @property
    def etype(self) -> str:
        """Return the entity type code (e.g., 'Q', 'CAT', etc.)."""
        try:
            part = self.flexo_id.split('-', 1)[1]
            return ''.join(filter(str.isalpha, part.split('-')[0]))  # up to first dash
        except IndexError:
            raise ValueError(f"Malformed Flex-O ID: {self.flexo_id}")

    @property
    def date_str(self) -> str:
        """Return the YYMMDD creation date as string."""
        try:
            part = self.flexo_id.split('-', 1)[1]
            # e.g. "Q251019" → skip type prefix, take next 6 digits
            digits = ''.join(ch for ch in part if ch.isdigit())
            return digits[:6]
        except IndexError:
            raise ValueError(f"Malformed Flex-O ID: {self.flexo_id}")

    @property
    def date(self) -> datetime:
        """Return the creation date as datetime.date object (UTC, naive)."""
        try:
            ds = self.date_str
            return datetime.strptime(ds, "%y%m%d").date()
        except Exception as e:
            raise ValueError(f"Invalid date in Flex-O ID: {self.flexo_id}") from e

    @property
    def hash_part(self) -> str:
        """Return the 6-hex BLAKE hash portion (e.g., '9B3E2')."""
        try:
            after_dash = self.flexo_id.split('-', 2)[2]
            return after_dash.split('@')[0]
        except IndexError:
            raise ValueError(f"Malformed Flex-O ID: {self.flexo_id}")

    @property
    def suffix(self) -> str:
        """Return the full suffix after '@' (e.g., '001A')."""
        try:
            return self.flexo_id.split('@', 1)[1]
        except IndexError:
            raise ValueError(f"Malformed Flex-O ID: {self.flexo_id}")

    @property
    def version(self) -> int:
        try:
            return int(self.suffix[:-1])  # drop state suffix
        except (ValueError, IndexError):
            return 1

    @property
    def prefix(self) -> str:
        # nur bis einschließlich Hash-Teil
        return self.flexo_id.split('@', 1)[0]
        # return self.flexo_id.split('@')[0].rsplit('-', 1)[0]

    def parsed(self) -> dict:
        """Return a structured breakdown of the Flex-O ID."""
        return {
            "domain": self.domain,
            "etype": self.etype,
            "date": self.date,
            "hash": self.hash_part,
            "version": self.version,
            "state": self.state_code,
        }
    # ──────────────────────────────────────────────────────────────────────────

    @classmethod
    def next_version(cls, oid) -> str:
        """
        Create the next version in the same ID lineage.

        Increments the version counter of an existing FlexOID while preserving
        its prefix. Used when an entity transitions to a
        new revision within the same lifecycle (e.g., minor updates or approvals).

        Parameters
        ----------
        oid : FlexOID
        The existing ID whose version is to be incremented.

        Returns
        -------
        FlexOID
        A new Flex-O ID with the same prefix, but version +1.

        Raises
        ------
        RuntimeError
        If the maximum allowed version (`MAX_VERSION`) is exceeded.

        Notes
        -----
        - Warnings are logged when the version approaches obsolescence.
        """
        new_ver = oid.version + 1

        if new_ver > cls.WARN_THRESHOLD and new_ver < cls.MAX_VERSION:
            logger.warning(f"{oid} approaching obsolescence ({new_ver}/999).")
        if new_ver > cls.MAX_VERSION:
            raise RuntimeError(f"{oid} exceeded {cls.MAX_VERSION}; mark obsolete.")

        new_id = f"{oid.prefix}@{new_ver:03d}{oid.state_code}"
        return cls(new_id)

    # ──────────────────────────────────────────────────────────────────────────
    @staticmethod
    def clone_new_base(domain: str, etype: str, estate: str, text: str):
        """
        Start a new Flex-O ID lineage for a derived or duplicated entity.

        This helper creates a completely new base ID (version 1) using the given
        parameters, instead of incrementing an existing version chain. It is used
        when an entity is copied, forked, or conceptually replaced by a new one.

        Returns
        -------
        FlexOID
        A new base ID starting at version 1, unrelated to the original lineage.

        Notes
        -----
        - Equivalent to calling `generate(..., version=1)` explicitly.
        - Used when creating "clones" or "variants" that should not share version history.
        """
        return FlexOID.safe_generate(domain, etype, estate, text, version=1)

    def __str__(self):
        return self.flexo_id

    def __repr__(self):
        return f"<FlexOID {self.flexo_id}>"

