import pytest

from flexoentity import FlexOID, FlexoEntity, EntityType, EntityState 

def test_initial_state(entity):
    assert entity.state == EntityState.DRAFT
    assert entity.flexo_id.version == 1
    assert len(entity.flexo_id.signature) == 16  # blake2s digest_size=8 → 16 hex
    assert FlexoEntity.verify_integrity(entity)


def test_approval_bumps_version(entity):
    entity.approve()
    assert entity.state == EntityState.APPROVED
    assert entity.flexo_id.version == 2


def test_signing_bumps_version(entity):
    entity.approve()
    v_before = entity.flexo_id
    entity.sign()
    assert entity.state == EntityState.APPROVED_AND_SIGNED
    assert entity.flexo_id != v_before


def test_publish_bumps_version(entity):
    entity.approve()
    entity.sign()
    v_before = entity.flexo_id.version
    entity.publish()
    assert entity.state == EntityState.PUBLISHED
    assert entity.flexo_id.version == v_before + 1


def test_modify_content_changes_fingerprint(entity):
    old_signature = entity.flexo_id.signature
    entity._seed = "Rephrased content"  # simulate text change
    entity._update_fingerprint()
    assert entity.flexo_id.signature != old_signature


def test_no_version_bump_on_draft_edits(entity):
    entity._seed = "Draft edit only"
    entity._update_fingerprint()
    assert entity.flexo_id.version == 1


def test_version_bump_after_edit_and_sign(entity):
    entity.approve()
    v1 = entity.flexo_id
    entity._seed = "Changed content"
    entity.sign()
    assert entity.flexo_id != v1


def test_integrity_check_passes_and_fails(entity):
    entity.approve()
    assert FlexoEntity.verify_integrity(entity)
    # simulate tampering
    entity._seed = "Tampered text"
    assert not FlexoEntity.verify_integrity(entity)


def test_obsolete_state(entity):
    entity.approve()
    entity.sign()
    entity.publish()
    entity.obsolete()
    assert entity.state == EntityState.OBSOLETE


def test_clone_new_base_resets_lineage(entity):
    entity.approve()
    entity.sign()
    entity.publish()
    entity.obsolete()
    old_id = entity.flexo_id
    entity.clone_new_base()
    assert entity.flexo_id != old_id
    assert entity.state == EntityState.DRAFT
    assert entity.flexo_id.version == 1


def test_mass_version_increments_until_obsolete(entity):
    entity.approve()
    for _ in range(FlexOID.MAX_VERSION - 2):
        entity.sign()
    with pytest.raises(RuntimeError, match="mark obsolete"):
        entity.sign()
