"""
flexoentity.collection
----------------------

A minimal collection for FlexOEntities.

- No domain rules
- No uniqueness validation (last write wins)
- No state or lifecycle logic
- No typing constraints

The goal is to provide a Smalltalk-like Collection object with a simple
protocol for storing, retrieving, iterating, and serializing FlexOEntities.
"""


from .flexo_entity import FlexoEntity
from .id_factory import FlexOID


class FlexoCollection:
    """
    A minimal collection of FlexOEntities, keyed by FlexOID.

    Examples:
        coll = FlexoCollection()
        coll.add(entity)
        entity = coll.get(some_id)
        for e in coll:
            ...
    """

    def __init__(self, items=None):
        self._items = {}

        if items:
            for it in items:
                self.add(it)

    def add(self, entity: FlexoEntity):
        """Add or replace an entity by its FlexOID. Overwrites on duplicate IDs."""
        self._items[entity.flexo_id] = entity

    def remove(self, oid: FlexOID):
        """Remove an entity by ID, ignoring if missing."""
        self._items.pop(oid, None)

    def get(self, oid: FlexOID):
        """Return the entity or None."""
        return self._items.get(oid)

    def __contains__(self, oid: FlexOID):
        """Check if an ID exists."""
        return oid in self._items

    def __len__(self) -> int:
        return len(self._items)

    def __iter__(self):
        return iter(self._items.values())

    def entities(self):
        """Return all entities."""
        return list(self._items.values())

    def ids(self):
        """Return all IDs."""
        return list(self._items.keys())

    # ------------------------------------------------------------------
    # Smalltalk-inspired interface
    # ------------------------------------------------------------------

    def at(self, oid):
        """Smalltalk-style accessor (raises if missing)."""
        return self._items[oid]

    def at_put(self, oid, entity):
        """Smalltalk-style setter."""
        self._items[oid] = entity

    def includes_id(self, oid: FlexOID):
        """Smalltalk-style membership test."""
        return oid in self._items

    def do(self, block):
        """Smalltalk-style #do: iteration."""
        for value in self._items.values():
            block(value)

    # ------------------------------------------------------------------
    # Serialization helpers
    # ------------------------------------------------------------------

    def to_dict_list(self):
        """Serialize entities to a list of dicts."""
        return [e.to_dict() for e in self._items.values()]

    @classmethod
    def from_dict_list(cls, data):
        """
        Deserialize a list of dicts into a FlexoCollection.

        Uses FlexoEntity.from_dict to dispatch to the correct subclass.
        """
        c = cls()
        for d in data:
            entity = FlexoEntity.from_dict(d)
            c.add(entity)
        return c

    def __repr__(self) -> str:
        return f"<FlexoCollection size={len(self._items)}>"
