import json
from .persistance_backend import PersistanceBackend


class SQLiteEntityBackend(PersistanceBackend):
    """
    SQLite backend storing and returning **dicts only**.

    Note:
      - EntityManager performs dict <-> entity conversion.
      - This backend does not call entity_class.from_dict().
    """

    def __init__(self, entity_class, conn, table_name):
        super().__init__(entity_class)
        self.conn = conn
        self.table = table_name
        self._init_schema()

    def _init_schema(self):
        self.conn.execute(f"""
            CREATE TABLE IF NOT EXISTS {self.table} (
                flexo_id TEXT PRIMARY KEY,
                json     TEXT NOT NULL
            )
        """)
        self.conn.commit()

    # -------------------------
    # Helpers
    # -------------------------

    def _ensure_dict(self, entity_or_dict) -> dict:
        """
        Transitional helper:
          - Accept dict (preferred)
          - Accept FlexoEntity-like objects with .to_dict() (legacy)
        """
        if isinstance(entity_or_dict, dict):
            return entity_or_dict
        if hasattr(entity_or_dict, "to_dict"):
            return entity_or_dict.to_dict()
        raise TypeError(f"Expected dict or object with to_dict(), got {type(entity_or_dict).__name__}")

    def _flexo_id_from_dict(self, entity_dict: dict) -> str:
        try:
            return entity_dict["meta"]["flexo_id"]
        except Exception as e:
            raise KeyError("entity_dict must contain meta.flexo_id") from e

    # -------------------------
    # Writes (dict)
    # -------------------------

    def save(self, entity_dict):
        d = self._ensure_dict(entity_dict)
        fid = self._flexo_id_from_dict(d)
        self.conn.execute(
            f"INSERT OR REPLACE INTO {self.table} (flexo_id, json) VALUES (?, ?)",
            (fid, json.dumps(d))
        )
        self.conn.commit()

    def update(self, entity_dict):
        d = self._ensure_dict(entity_dict)
        fid = self._flexo_id_from_dict(d)
        self.conn.execute(
            f"UPDATE {self.table} SET json = ? WHERE flexo_id = ?",
            (json.dumps(d), fid)
        )
        self.conn.commit()

    def delete(self, flexo_id):
        self.conn.execute(
            f"DELETE FROM {self.table} WHERE flexo_id = ?",
            (flexo_id,)
        )
        self.conn.commit()

    # -------------------------
    # Reads (dict)
    # -------------------------

    def load(self, flexo_id):
        row = self.conn.execute(
            f"SELECT json FROM {self.table} WHERE flexo_id = ?",
            (flexo_id,),
        ).fetchone()

        if not row:
            return None

        # row indexing depends on connection row_factory; support both
        raw = row["json"] if isinstance(row, dict) or hasattr(row, "__getitem__") else row[0]
        if not isinstance(raw, str):
            raw = raw[0]
        return json.loads(raw)

    def load_all(self):
        rows = self.conn.execute(
            f"SELECT json FROM {self.table}"
        ).fetchall()

        result = []
        for r in rows:
            raw = r["json"] if isinstance(r, dict) or hasattr(r, "__getitem__") else r[0]
            if not isinstance(raw, str):
                raw = raw[0]
            result.append(json.loads(raw))
        return result

    def clear(self):
        self.conn.execute(f"DELETE FROM {self.table}")
        self.conn.commit()
