class PersistanceBackend:
    """
    Interface for all persistence backends.

    Contract (Option A):
      - Backends store serialized entity dictionaries.
      - Backends return dictionaries (never FlexoEntity instances).
      - EntityManager is responsible for dict <-> entity conversion.

    Stored dict format must include:
      - meta.flexo_id (string)
      - meta.schema.name / meta.schema.version
      - content (entity-specific)
    """

    def __init__(self, entity_class):
        self._entity_class = entity_class

    @property
    def entity_class(self):
        return self._entity_class

    @entity_class.setter
    def entity_class(self, a_class):
        self._entity_class = a_class

    # -------------------------
    # Write API (dict)
    # -------------------------

    def save(self, entity_dict: dict) -> None:
        raise NotImplementedError

    def update(self, entity_dict: dict) -> None:
        raise NotImplementedError

    def delete(self, flexo_id: str) -> None:
        raise NotImplementedError

    # -------------------------
    # Read API (dict)
    # -------------------------

    def load(self, flexo_id: str) -> dict | None:
        raise NotImplementedError

    def load_all(self) -> list[dict]:
        raise NotImplementedError

    def clear(self):
        """Remove all stored entities."""
        raise NotImplementedError
