import json
from .persistance_backend import PersistanceBackend
from .in_memory_backend import InMemoryBackend


class JsonFileBackend(PersistanceBackend):
    """
    JSON-file based backend.

    Uses an internal InMemoryBackend and syncs to a single JSON file on disk.
    Dict-only contract (Option A).
    """

    def __init__(self, entity_class, path):
        super().__init__(entity_class=entity_class)
        self._mem = InMemoryBackend(entity_class)
        self._path = path

    # core API delegates (dicts)

    def save(self, entity_dict: dict):
        self._mem.save(entity_dict)

    def update(self, entity_dict: dict):
        self._mem.update(entity_dict)

    def delete(self, flexo_id: str):
        self._mem.delete(flexo_id)

    def load(self, flexo_id: str):
        return self._mem.load(flexo_id)

    def load_all(self):
        return self._mem.load_all()

    def clear(self):
        self._mem.clear()

    # file sync (dicts)

    def flush_to_file(self):
        data = self._mem.load_all()
        with open(self._path, "w", encoding="utf-8") as f:
            json.dump(data, f, ensure_ascii=False, indent=2)

    def load_from_file(self):
        try:
            with open(self._path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except FileNotFoundError:
            return

        self._mem.clear()
        for d in data:
            self._mem.save(d)
