import base64
from dataclasses import dataclass
from typing import Optional
from uuid import UUID
from flexoentity import FlexoEntity, FlexOID, EntityType


@dataclass
class CertificateReference:
    """
    A cross-platform reference to a signing certificate.

    This does NOT contain private keys or secret material.
    It only contains the information required for OS-provided APIs
    (OpenSSL, certutil, security) to *locate* or *verify* the certificate.

    Fields:
      platform:
         Optional override for backend creation.
         Values: "LINUX", "WINDOWS", "MACOS" (case-insensitive).
         If None, platform.system() determines the backend.

      identifier:
         Linux:   path to certificate PEM file
         Windows: certificate thumbprint (SHA1, with or without colons)
         macOS:   certificate Common Name (CN) in Keychain

      private_key_path:
         Linux only:
            path to private key PEM file.
         Windows/macOS:
            MUST be None.

      public_cert_path:
         The path to the public certificate for verification.
         Linux:
            optional (defaults to `identifier`)
         Windows:
            optional; used if you want to validate against a specific cert
         macOS:
            REQUIRED (used for OpenSSL verification)
    """

    platform: Optional[str] = None
    identifier: str = ""
    private_key_path: Optional[str] = None
    public_cert_path: Optional[str] = None

    def to_dict(self):
        return {
            "platform": self.platform,
            "identifier": self.identifier,
            "private_key_path": self.private_key_path,
            "public_cert_path": self.public_cert_path
        }


class FlexoSignature(FlexoEntity):
    ENTITY_TYPE = EntityType.ATTESTATION

    def __init__(
        self,
        *,
        flexo_id,
        signed_entity: Optional[FlexOID] = None,
        signer_id: Optional[UUID] = None,
        signature_data: str = "",
        signature_type: str = "PKCS7-DER",
        certificate_reference: CertificateReference | None = None,
        certificate_thumbprint: str = "",
        comment: Optional[str] = None,
        subtype=None,
        fingerprint=None,
        origin=None,
        originator_id=None,
        owner_id=None,
    ):
        # content fields
        self.signed_entity = signed_entity
        self.signer_id = signer_id
        self.signature_data = signature_data
        self.signature_type = signature_type
        self.certificate_reference = certificate_reference
        self.certificate_thumbprint = certificate_thumbprint
        self.comment = comment

        # meta fields
        super().__init__(
            flexo_id=flexo_id,
            subtype=subtype,
            fingerprint=fingerprint,
            origin=origin,
            originator_id=originator_id,
            owner_id=owner_id,
        )

    def _serialize_content(self):
        return {
            "signed_entity": str(self.signed_entity) if self.signed_entity else None,
            "signer_id": str(self.signer_id) if self.signer_id else None,
            "signature_data": self.signature_data,
            "signature_type": self.signature_type,
            "certificate_thumbprint": self.certificate_thumbprint,
            "comment": self.comment,
        }

    @property
    def text_seed(self):
        return f"{self.signed_entity}:{self.signer_id}:{self.certificate_thumbprint}"

    @classmethod
    def default(cls):
        """
        Required by FlexoEntity.
        Returns an empty draft signature.
        """
        return cls.with_domain_id(domain_id="GEN")

    @classmethod
    def create_signed(cls, data: bytes, entity: FlexOID, signer_id: UUID, backend):
        sig = backend.sign(data)

        return cls.with_domain_id(
            domain_id=entity.domain_id,
            signed_entity=entity,
            signer_id=signer_id,
            signature_data=base64.b64encode(sig).decode(),
            certificate_reference=backend.cert_ref,
            certificate_thumbprint=backend.certificate_thumbprint,
        )

    def verify(self, data: bytes, backend) -> bool:
        raw = base64.b64decode(self.signature_data)
        return backend.verify(data, raw)
