from .flexo_entity import FlexoEntity
from .typed_collection import TypedCollection


class EntityNotFoundError(KeyError):
    pass


class InvalidEntityError(ValueError):
    pass


class DuplicateEntityError(ValueError):
    pass


class EntityManager:
    """
    Backend-agnostic manager for any FlexoEntity subclass.

    Option A contract:
      - backends store/return dicts only
      - manager converts dict <-> entity
    """

    ENTITY_CLASS = None  # must be overridden by subclasses

    def __init__(self, local_backend, staging_backend, permanent_backend):
        if self.ENTITY_CLASS is None:
            raise ValueError("Subclasses must define ENTITY_CLASS")

        if not issubclass(self.ENTITY_CLASS, FlexoEntity):
            raise TypeError("ENTITY_CLASS must be a subclass of FlexoEntity")

        self.local_backend = local_backend
        self.staging_backend = staging_backend
        self.permanent_backend = permanent_backend

    # ------------------------------------------------------------------
    # Conversion helpers
    # ------------------------------------------------------------------

    def _to_dict(self, entity) -> dict:
        self._ensure_type(entity)
        return entity.to_dict()

    def _to_entity(self, entity_dict: dict):
        if entity_dict is None:
            return None
        return self.ENTITY_CLASS.from_dict(entity_dict)

    def _to_entities(self, dicts: list[dict]):
        return [self._to_entity(d) for d in dicts]

    # ------------------------------------------------------------------
    # Locators
    # ------------------------------------------------------------------

    def backend_of_domain_id(self, domain_id: str):
        # Note: this converts dict->entity because domain_id is a FlexoEntity property
        for backend_name, backend in [
            ("local", self.local_backend),
            ("staging", self.staging_backend),
            ("permanent", self.permanent_backend),
        ]:
            for d in backend.load_all():
                e = self._to_entity(d)
                if e.domain_id == domain_id:
                    return backend_name
        return None

    def backend_of_flexo_id(self, flexo_id: str):
        for backend_name, backend in [
            ("local", self.local_backend),
            ("staging", self.staging_backend),
            ("permanent", self.permanent_backend),
        ]:
            if backend.load(flexo_id) is not None:
                return backend_name
        return None

    # ------------------------------------------------------------------
    # CRUD operations (entity API)
    # ------------------------------------------------------------------

    def add(self, entity):
        self.local_backend.save(self._to_dict(entity))

    def update(self, entity):
        self.local_backend.update(self._to_dict(entity))

    def delete(self, flexo_id: str):
        self.local_backend.delete(flexo_id)

    # ------------------------------------------------------------------
    # Retrieval (entity API)
    # ------------------------------------------------------------------

    def get(self, flexo_id: str):
        d = self.local_backend.load(flexo_id)
        return self._to_entity(d)

    # FIXME: Readd staging backend later
    def all(self):
        dicts = (
            self.local_backend.load_all()
            # + self.staging_backend.load_all()
            + self.permanent_backend.load_all()
        )
        return self._to_entities(dicts)

    def promote_to_staging(self, flexo_id):
        d = self.local_backend.load(flexo_id)
        if d is None:
            raise EntityNotFoundError(flexo_id)
        self.staging_backend.save(d)
        self.local_backend.delete(flexo_id)

    def promote_to_permanent(self, flexo_id):
        d = self.staging_backend.load(flexo_id)
        if d is None:
            raise EntityNotFoundError(flexo_id)
        self.permanent_backend.save(d)
        self.staging_backend.delete(flexo_id)

    def sync_all(self):
        # sync staging → local
        for d in self.staging_backend.load_all():
            self.local_backend.save(d)

        # sync permanent → local
        for d in self.permanent_backend.load_all():
            self.local_backend.save(d)

        # NOTE: refresh() is not defined here.
        # If you want a hook, define it explicitly, otherwise remove this call.
        if hasattr(self, "refresh"):
            self.refresh()

    # ------------------------------------------------------------------
    # Helpers
    # ------------------------------------------------------------------

    def exists(self, flexo_id: str) -> bool:
        return self.local_backend.load(flexo_id) is not None

    def count(self) -> int:
        return len(self.local_backend.load_all())

    def add_or_update(self, entity):
        fid = str(entity.flexo_id)
        if self.exists(fid):
            self.update(entity)
        else:
            self.add(entity)

    def as_collection(self):
        # Collection expects entities (your current TypedCollection usage suggests this)
        return TypedCollection(self.ENTITY_CLASS, items=self.all())

    # ------------------------------------------------------------------
    # Internal
    # ------------------------------------------------------------------

    def _ensure_type(self, entity):
        if not isinstance(entity, self.ENTITY_CLASS):
            raise TypeError(
                f"Expected {self.ENTITY_CLASS.__name__}, got {type(entity).__name__}"
            )

    def __repr__(self):
        return f"<EntityManager for {self.ENTITY_CLASS.__name__}>"
