from typing import Dict, Optional
from flexoentity import FlexOID, EntityType, EntityState
from .domain import Domain


class DomainManager:
    """
    I manage all Domain instances in the system.

    Responsibilities:
        • ensure unique domain codes
        • provide lookup by code and by FlexOID
        • restore domains from dict/json safely
        • serve as the canonical source of domain objects
    """

    _by_code: Dict[str, Domain] = {}
    _by_oid: Dict[str, Domain] = {}

    # ---------------------------------------------------------------
    # Registration
    # ---------------------------------------------------------------
    @classmethod
    def register(cls, domain: Domain):
        """Register a domain; raise error if code already exists."""
        code = domain.domain_id
        oid = str(domain.flexo_id)

        if code in cls._by_code:
            raise ValueError(f"Domain code already registered: {code}")

        cls._by_code[code] = domain
        cls._by_oid[oid] = domain
        return domain

    @classmethod
    def register_or_replace(cls, domain: Domain):
        """Replace an existing domain with same code — rarely useful, but optional."""
        code = domain.code
        oid = str(domain.flexo_id)
        cls._by_code[code] = domain
        cls._by_oid[oid] = domain
        return domain

    @classmethod
    def get(cls, code: str) -> Domain:
        """Return the domain for given code; error if missing."""
        if code not in cls._by_code:
            raise KeyError(f"Unknown domain code: {code}")
        return cls._by_code[code]

    # ---------------------------------------------------------------
    # Lookup helpers
    # ---------------------------------------------------------------
    @classmethod
    def get_by_oid(cls, oid) -> Domain:
        """Return the domain with the given FlexOID string or object."""
        key = str(oid)
        if key not in cls._by_oid:
            raise KeyError(f"Unknown domain OID: {key}")
        return cls._by_oid[key]

    # ---------------------------------------------------------------
    # Creation helpers
    # ---------------------------------------------------------------
    @classmethod
    def create(cls, domain_id: str, **kwargs) -> Domain:
        """
        Create a new domain, register it, and return it.
        Raises if code exists.
        """
        flexo_id = FlexOID.safe_generate(domain_id=domain_id,
                                         entity_type=EntityType.DOMAIN.value,
                                         state=EntityState.DRAFT.value, text=domain_id)
        domain = Domain(flexo_id=flexo_id, **kwargs)
        return cls.register(domain)

    # ---------------------------------------------------------------
    # Utility
    # ---------------------------------------------------------------
    @classmethod
    def list(cls):
        """Return all domain objects."""
        return list(cls._by_code.values())

    @classmethod
    def all_domain_ids(cls):
        return list(cls._by_code.keys())

    @classmethod
    def clear(cls):
        """Clear registry — useful for tests."""
        cls._by_code.clear()
        cls._by_oid.clear()
